﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Parsing;
using ParserGenerator;
using System.IO;

namespace TrafficLightController
{
    /// <summary>
    /// The implementation of the application-specific
    /// parts of the state machine controller.
    /// </summary>
    
    public class CrossingController : FSM, IEnumerable<IToken>
    {
        /// <summary>
        /// Factory method
        /// </summary>
        /// <returns>An instance of a crossing controller</returns>
        
        public static CrossingController Create(TextWriter debugStream, TextWriter errStream)
        {
            using (StreamReader grammarStream = new StreamReader("CrossingController.g"))
            {
                FSMFactory<CrossingController> ff = new FSMFactory<CrossingController>(grammarStream, false);
                CrossingController cc = ff.CreateInstance();
                return cc;
            }
        }

        /// <summary>
        /// The traffic light being controlled
        /// </summary>
        
        public TrafficLight Light
        {
            get;
            private set;
        }

        /// <summary>
        /// The pedestrian walk indicator being controlled
        /// </summary>
        
        public StopWalkIndicator WalkIndicator
        {
            get;
            private set;
        }

        private TimerQueue crossingTimer;

        /// <summary>
        /// Create a crossing controller that starts
        /// life permitting vehicles to pass, and
        /// telling pedestrians to wait.
        /// </summary>
        
        public CrossingController()
        {
            Light = new TrafficLight();
            WalkIndicator = new StopWalkIndicator();
            WalkIndicator.ButtonPressed = false;
            crossingTimer = new TimerQueue();
            inputEvents = new Queue<IToken>();
        }

	    // Actions that set the colour of the
	    // traffic lights, or the crossing indicator.

	    public void SetLightRedAndYellow()
        {
            Light.Colour = LightColour.REDANDYELLOW;
        }

	    public void SetLightGreen()
        {
            Light.Colour = LightColour.GREEN;
        }

	    public void SetLightRed()
        {
            Light.Colour = LightColour.RED;
        }

	    public void SetLightYellow()
        {
            Light.Colour = LightColour.YELLOW;
        }

	    public void SetLightWalk()
        {
            WalkIndicator.CanWalk = true;
        }

	    public void SetLightDontWalk()
        {
            WalkIndicator.CanWalk = false;
        }

	    // Action to capture the fact that the
	    // request to cross button has been pressed.

	    public void RecordButtonPressed()
        {
            WalkIndicator.ButtonPressed = true;
        }

	    // Actions to queue a time interval for the
	    // various phases of the crossing controller.

	    public void SetGreenTimer()
        {
            crossingTimer.Schedule(DateTime.Now.AddSeconds(3), InjectTickEvent);
        }

	    public void SetYellowTimer()
        {
            crossingTimer.Schedule(DateTime.Now.AddSeconds(3), InjectTickEvent);
        }

	    public void SetWalkTimer()
        {
            crossingTimer.Schedule(DateTime.Now.AddSeconds(3), InjectTickEvent);
        }

	    public void SetAllRedTimer()
        {
            crossingTimer.Schedule(DateTime.Now.AddSeconds(3), InjectTickEvent);
        }

	    public void SetRedAndYellowTimer()
        {
            crossingTimer.Schedule(DateTime.Now.AddSeconds(3), InjectTickEvent);
        }

	    // Clear down the record of the request to
	    // cross button, as the crossing has been
	    // permitted.

        public void ClearButtonPress()
        {
            WalkIndicator.ButtonPressed = false;
        }

        // Event input management

        private Queue<IToken> inputEvents;

        /// <summary>
        /// Cause a timer tick event to be placed on the input
        /// </summary>
        
        public void InjectTickEvent()
        {
            ProcessEvent(new ParserToken(FSMTable.Tokens["TIMERTICK"]));
        }

        /// <summary>
        /// Cause a button press event to be placed on the input
        /// </summary>
        
        public void InjectButtonPress()
        {
            ProcessEvent(new ParserToken(FSMTable.Tokens["BUTTONPRESS"]));
        }

        /// <summary>
        /// Capture the next input event from the input queue. Note that
        /// this uses a polled loop to capture the event. A better approach
        /// would be to use event objects on the queue.
        /// </summary>
        /// <returns>Each successive input token</returns>
        
        public IEnumerator<IToken> GetEnumerator()
        {
            while (inputEvents.Count <= 0)
                System.Threading.Thread.Sleep(500);
            yield return inputEvents.Dequeue();
        }

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            while (inputEvents.Count <= 0)
                System.Threading.Thread.Sleep(500);
            yield return inputEvents.Dequeue();
        }
    }
}
