﻿options
{
    using Parsing,
    namespace CalculatorDemo,
    parserclass GLRCalculator
}

tokens
{
    LPAREN,    RPAREN,    PLUS,    
    MINUS,     TIMES,     DIVIDE,    
    POWER,     NUMBER <string>,  
    PERIOD,    EXPONENT
}

grammar(result)
{
    result:
        expression
        {
            // Result is a property of the GLRCalculator 
            // parser class, used to extract the result
            // of the calculation after the parser has run.

            Result = $0.ToString();
        }
    ;

	leafExpression <double> :
	    optMinus number
        {
            $$ = $0 * $1;
        }

	|   optMinus LPAREN expression RPAREN
        {
            $$ = $0 * $2;
        }
	;

	optMinus <double> :
		MINUS
		{
			$$ = -1.0;
		}
	|
		{
			$$ = 1.0;
		}
	;

    expression <double> :
        leafExpression
        {	$$ = $0;	}
    |   expression POWER expression
        {	$$ = Math.Pow($0, $2);	}
    |   expression TIMES expression
        {   $$ = $0 * $2;  }

    |   expression DIVIDE expression
        {	$$ = $0 / $2;  }
    |   expression PLUS expression
        {   $$ = $0 + $2;  }
    |   expression MINUS expression
        {   $$ = $0 - $2;  }

    merge
        {
            // Merge only needs to deal with situations
            // where we have expression OP expression
            // in which one of the children is also an
            // expression OP expression. In this case,
            // the job of the GLR merge is to choose
            // the interpretation that gives correct
            // operator precedence.

            if(IsExprOpExpr($0))
            {
                var r = $0.Children[2] as NonterminalToken;
                var l = $0.Children[0] as NonterminalToken;

                if(IsExprOpExpr(r))
                {
                    // Example where swap must take place:
                    // $0
                    //  |
                    // Expr TIMES Expr
                    //             |
                    //            Expr PLUS Expr
                    //
                    // Which should be replaced by the other parse of the
                    // same sequence of input tokens which will look like:
                    // $1
                    //  |
                    // Expr PLUS Expr
                    //  |
                    // Expr TIMES Expr

                    if(HigherPrecedence($0.Children[1], r.Children[1]))
                        $$ = $1;
                    else
                        $$ = $0;
                }
                else if(IsExprOpExpr(l))
                {
                    // Example where the following swap must take place:
                    // $0
                    //  |
                    // Expr TIMES Expr
                    //  |
                    // Expr PLUS Expr
                    //
                    // Which should be replaced by the other parse of the
                    // same sequence of input tokens which will look like:
                    // $1
                    //  |
                    // Expr PLUS Expr
                    //            |
                    //           Expr TIMES Expr

                    if(HigherPrecedence($0.Children[1], l.Children[1]))
                        $$ = $1;
                    else
                        $$ = $0;
                }
            }
        }
    ;

    number <double> :
        NUMBER exponent
        {
            $$ = double.Parse($0) * Math.Pow(10.0, $1);
		}

    |   NUMBER mantissa exponent
        {
            $$ = (double.Parse($0) + $1) 
				* Math.Pow(10.0, $2);
        }

	|	mantissa exponent
		{
            $$ = $0 * Math.Pow(10.0, $1);
        }
    ;

	mantissa <double> : 
        PERIOD NUMBER
		{
			int digitCount = $1.Length;
            double mantissa = double.Parse($1);
            while(--digitCount >= 0)
                mantissa *= 0.1;
			$$ = mantissa;
		}
	;

    exponent <double> :
        EXPONENT NUMBER
        {
            $$ = double.Parse($1);
        }

    |   EXPONENT MINUS NUMBER
        {
            $$ = - double.Parse($2);
        }

    |
        {
            $$ = 0.0;
        }
    ;
}