﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using TrafficLightController;
using System.IO;
using Parsing;
using ParserGenerator;

namespace TrafficLightDemo
{
    public partial class Form1 : Form
    {
        private CrossingController crossingController;
        private StringBuilder debugText;
        private StringBuilder errText;

        public Form1()
        {
            InitializeComponent();

            // Create an instance of a crossing controller,
            // with its embedded grammar-driven state machine

            debugText = new StringBuilder();
            errText = new StringBuilder();

            // To build an inline state machine from the grammar input file
            // at runtime, uncomment the following statement. Comment out
            // the subsequent two statements that creates an offline parser.

            // crossingController = CrossingController.Create
            //     (new StringWriter(debugText), new StringWriter(errText));

            // Code to create an offline state machine, using source code
            // as generated from the input grammar using PARSELR

            FSMFactory<CrossingController> ff = new FSMFactory<CrossingController>();
            crossingController = ff.CreateInstance();

            if (crossingController != null)
            {
                crossingController.ErrStream = new StringWriter(errText);
                crossingController.DebugStream = new StringWriter(debugText);
                crossingController.Light.PropertyChanged 
                    += new EventHandler(Model_Changed);
                crossingController.WalkIndicator.PropertyChanged 
                    += new EventHandler(Model_Changed);
            }
        }

        private void Form1_Paint(object sender, PaintEventArgs e)
        {
            // Draw the traffic light, then the pedestrian crossing panel.
            // Lastly, render state transition messages into the output text box.

            if (crossingController != null)
            {
                RenderTrafficLight(crossingController.Light, e.Graphics);
                RenderCrossingIndicator(crossingController.WalkIndicator, e.Graphics);
                txtDebug.Text = debugText.ToString();
            }
            else
                txtDebug.Text = errText.ToString();

            // Cause the text box to scroll to the bottom of its text.

            txtDebug.SelectionStart = txtDebug.Text.Length;
            txtDebug.SelectionLength = 0;
            txtDebug.ScrollToCaret();
        }

        private void RenderCrossingIndicator(StopWalkIndicator walkIndicator, Graphics g)
        {
            // Establish shape positions and sizes

            Point indicatorOffset = new Point(200, 300);
            Size indicatorSize = new Size(100, 120);
            Rectangle indicatorRect = new Rectangle(indicatorOffset, indicatorSize);
            Rectangle walkRect = new Rectangle
            (
                indicatorRect.Left,
                indicatorRect.Top + indicatorRect.Height / 4,
                indicatorRect.Width,
                indicatorRect.Height / 4
            );
            Rectangle dontWalkRect = new Rectangle
            (
                indicatorRect.Left,
                indicatorRect.Top,
                indicatorRect.Width,
                indicatorRect.Height / 4
            );
            Rectangle waitRect = new Rectangle
            (
                indicatorRect.Left,
                indicatorRect.Top + indicatorRect.Height / 2,
                indicatorRect.Width,
                indicatorRect.Height / 4
            );

            Size poleSize = new Size(16, 220);
            Point poleTLHC = new Point(
                indicatorRect.Left + (indicatorRect.Width - poleSize.Width) / 2,
                280);
            Rectangle poleRect = new Rectangle(poleTLHC, poleSize);

            // Render the walk signal

            g.FillRectangle(Brushes.Gray, poleRect);
            g.FillRectangle(Brushes.Black, indicatorRect);
            StringFormat fmt = new StringFormat();
            fmt.Alignment = StringAlignment.Center;
            fmt.LineAlignment = StringAlignment.Center;

            Font indicatorFont = new Font("Arial Narrow", 12, FontStyle.Bold);
            if(walkIndicator.CanWalk)
                g.DrawString("WALK", indicatorFont, Brushes.LimeGreen, walkRect, fmt);
            else
                g.DrawString("DON'T WALK", indicatorFont, Brushes.Red, dontWalkRect, fmt);
            if (walkIndicator.ButtonPressed)
                g.DrawString("WAIT", indicatorFont, Brushes.Ivory, waitRect, fmt);
        }

        private void RenderTrafficLight(TrafficLight light, Graphics g)
        {
            // Establish shape positions and sizes

            Point lightOffset = new Point(100, 100);
            Size lightSize = new Size(60, 180);
            Rectangle lightRect = new Rectangle(lightOffset, lightSize);
            Size poleSize = new Size(16, 420);
            Point poleTLHC = new Point(
                lightRect.Left + (lightRect.Width - poleSize.Width) / 2,
                lightRect.Top - 20);
            Rectangle poleRect = new Rectangle(poleTLHC, poleSize);
            Size lampSize = new Size(40, 40);
            Rectangle redCircleRect = new Rectangle(
                lightRect.Left + (lightRect.Width - lampSize.Width) / 2,
                lightRect.Top + 10, lampSize.Width, lampSize.Height);
            Rectangle yellowCircleRect = new Rectangle(
                lightRect.Left + (lightRect.Width - lampSize.Width) / 2,
                lightRect.Top + 70, lampSize.Width, lampSize.Height);
            Rectangle greenCircleRect = new Rectangle(
                lightRect.Left + (lightRect.Width - lampSize.Width) / 2,
                lightRect.Top + 130, lampSize.Width, lampSize.Height);
            
            // Now paint the lights
            
            g.FillRectangle(Brushes.Gray, poleRect);
            g.FillRectangle(Brushes.Black, lightRect);
            if (light.Red)
                g.FillEllipse(Brushes.Red, redCircleRect);
            else
                g.FillEllipse(Brushes.DarkGray, redCircleRect);
            if(light.Yellow)
                g.FillEllipse(Brushes.Yellow, yellowCircleRect);
            else
                g.FillEllipse(Brushes.DarkGray, yellowCircleRect);
            if(light.Green)
                g.FillEllipse(Brushes.LimeGreen, greenCircleRect);
            else
                g.FillEllipse(Brushes.DarkGray, greenCircleRect);
        }

        private void Model_Changed(object sender, EventArgs e)
        {
            Invalidate();
        }

        private void btnCross_Click(object sender, EventArgs e)
        {
            // When the 'Cross' button os pressed, inject a
            // button press event into the state machine

            if(crossingController != null)
                crossingController.InjectButtonPress();
            Invalidate();
        }
    }
}
