﻿using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using SimpleInjector;
using SimpleInjector.Extensions;
using VersionLib;

namespace VersionLibTest
{
    [TestClass]
    public class VersionedTest
    {
        IVersionedFactory<string> vfs;

        [TestInitialize]
        public void Setup()
        {
            Container c = new Container();
            c.RegisterPackages();
            vfs = c.GetInstance<IVersionedFactory<string>>();
            Assert.IsNotNull(vfs);
        }

        [TestMethod]
        public void CreationTest()
        {
            IVersioned<string> vs = vfs.Create();
            Assert.IsNotNull(vs);
            Assert.IsFalse(vs.ExistedAt(DateTime.Now));
            IKeyWithValue<int> kwv = vs.ID as IKeyWithValue<int>;
            Assert.IsTrue(kwv.Value > 0);
        }

        [TestMethod]
        public void InsertAtTest()
        {
            IVersioned<string> vs = vfs.Create();
            Assert.IsNotNull(vs);
            vs.InsertAt(new DateTime(2013, 1, 3), "THIRD", false);
            vs.InsertAt(new DateTime(2013, 1, 1), "FIRST", false);
            vs.InsertAt(new DateTime(2013, 1, 2), "SECOND", false);
            Assert.IsTrue(vs.ExistedAt(DateTime.Now));
        }

        [TestMethod]
        public void ExistedAtTest()
        {
            IVersioned<string> vs = vfs.Create();
            Assert.IsNotNull(vs);
            vs.InsertAt(new DateTime(2013, 1, 3), "THIRD", false);
            vs.InsertAt(new DateTime(2013, 1, 1), "FIRST", false);
            vs.InsertAt(new DateTime(2013, 1, 2), "SECOND", false);
            Assert.IsTrue(vs.ExistedAt(DateTime.Now));
            Assert.IsTrue(vs.ExistedAt(new DateTime(2013, 1, 1, 12, 0, 0)));
            Assert.IsFalse(vs.ExistedAt(new DateTime(2012, 12, 31)));
        }

        [TestMethod]
        public void AtTest()
        {
            IVersioned<string> vs = vfs.Create();
            Assert.IsNotNull(vs);
            vs.InsertAt(new DateTime(2013, 1, 3), "THIRD", false);
            vs.InsertAt(new DateTime(2013, 1, 1), "FIRST", false);
            vs.InsertAt(new DateTime(2013, 1, 2), "SECOND", false);
            string s = vs.At(new DateTime(2012, 12, 31));
            Assert.IsNull(s);
            s = vs.At(new DateTime(2013, 1, 1, 0, 0, 0, 1));
            Assert.AreEqual("FIRST", s);
            s = vs.At(new DateTime(2013, 1, 2, 0, 0, 0, 1));
            Assert.AreEqual("SECOND", s);
            s = vs.At(new DateTime(2013, 1, 2, 23, 59, 59, 999));
            Assert.AreEqual("SECOND", s);
            s = vs.At(new DateTime(2013, 1, 3, 0, 0, 0, 1));
            Assert.AreEqual("THIRD", s);
            s = vs.At(DateTime.Now);
            Assert.AreEqual("THIRD", s);
        }

        [TestMethod]
        public void CopyAtTest()
        {
            IVersioned<string> vs = vfs.Create();
            Assert.IsNotNull(vs);
            vs.InsertAt(new DateTime(2013, 1, 3), "THIRD", false);
            vs.InsertAt(new DateTime(2013, 1, 1), "FIRST", false);
            vs.InsertAt(new DateTime(2013, 1, 2), "SECOND", false);
            string s = vs.At(new DateTime(2012, 12, 31));
            Assert.IsNull(s);
            s = vs.CopyAt(new DateTime(2013, 1, 1, 0, 0, 0, 1));
            Assert.AreEqual("FIRST", s);
            s = vs.CopyAt(new DateTime(2013, 1, 2, 0, 0, 0, 1));
            Assert.AreEqual("SECOND", s);
            s = vs.CopyAt(new DateTime(2013, 1, 2, 23, 59, 59, 999));
            Assert.AreEqual("SECOND", s);
            s = vs.CopyAt(new DateTime(2013, 1, 3, 0, 0, 0, 1));
            Assert.AreEqual("THIRD", s);
            s = vs.CopyAt(DateTime.Now);
            Assert.AreEqual("THIRD", s);
        }

        [TestMethod]
        public void ValueTest()
        {
            IVersioned<string> vs = vfs.Create();
            Assert.IsNotNull(vs);
            vs.InsertAt(new DateTime(2013, 1, 3), "THIRD", false);
            vs.InsertAt(new DateTime(2013, 1, 1), "FIRST", false);
            vs.InsertAt(new DateTime(2013, 1, 2), "SECOND", false);
            Assert.AreEqual("THIRD", vs.Value);
        }

        [TestMethod]
        public void CopyValueTest()
        {
            IVersioned<string> vs = vfs.Create();
            Assert.IsNotNull(vs);
            vs.InsertAt(new DateTime(2013, 1, 3), "THIRD", false);
            vs.InsertAt(new DateTime(2013, 1, 1), "FIRST", false);
            vs.InsertAt(new DateTime(2013, 1, 2), "SECOND", false);
            Assert.AreEqual("THIRD", vs.CopyValue());
        }
    }
}
