﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace VersionLib
{
    public interface IVersionedFactory<T>
    {
        /// <summary>
        /// Create a new empty versioned instance of type T. There
        /// is no current version of the object of type T after
        /// instantiation. The initial value will need to be inserted.
        /// </summary>
        /// <returns>A new empty versioned instance of a T</returns>

        IVersioned<T> Create();

        /// <summary>
        /// Find the versioned object with the specified key
        /// </summary>
        /// <param name="key">Key for versioned object</param>
        /// <returns>The object with the provided key, or null if not found</returns>

        IVersioned<T> Find(IKey key);

        /// <summary>
        /// If the object with the specified identity had an
        /// existing instance at the specified time, return
        /// the versioned object. Otherwise return null.
        /// </summary>
        /// <param name="key">The identity of the object being sought.</param>
        /// <param name="when">The time for which we are inspecting it.</param>
        /// <returns>The versioned object requested, or if it did not
        /// exist at that time, a null reference.</returns>

        IVersioned<T> FindIfExistedAt(IKey key, DateTime when);

        /// <summary>
        /// Find the specific unversioned object with the specified identity as
        /// it was at the stipulated time.
        /// </summary>
        /// <param name="key">The identity of the object being sought.</param>
        /// <param name="when">The time for which we are inspecting it.</param>
        /// <returns>The unversioned object requested, or if it did not
        /// exist at that time, a default value for its type.</returns>

        T FindAt(IKey key, DateTime when);
    }

    /// <summary>
    /// The set of all versioned objects of type T
    /// </summary>
    /// <typeparam name="T">The type of each versioned object in the class</typeparam>
    
    public class VersionedFactory<T> : IVersionedFactory<T>
    {
        private Dictionary<IKey, IVersioned<T>> objects;

        /// <summary>
        /// Create a new versioned set (equivalent to a class).
        /// </summary>
        /// <param name="keyGenerator">The component that generates
        /// unique keys for new versioned objects.</param>
        /// <param name="versionedGenerator">The component that
        /// generates new versioned objects.</param>
        
        public VersionedFactory()
        {
            objects = new Dictionary<IKey, IVersioned<T>>();
        }

        /// <summary>
        /// Create a new empty versioned instance of type T. There
        /// is no current version of the object of type T after
        /// instantiation. The initial value will need to be inserted.
        /// </summary>
        /// <returns>A new empty versioned instance of a T</returns>
        
        public IVersioned<T> Create()
        {
            IVersioned<T> newInstance = VersionLibPackage.Container
                .GetInstance<IVersioned<T>>();
            objects.Add(newInstance.ID, newInstance);
            return newInstance;
        }

        /// <summary>
        /// Find the versioned object with the specified key
        /// </summary>
        /// <param name="key">Key for versioned object</param>
        /// <returns>The object with the provided key, or null if not found</returns>

        public IVersioned<T> Find(IKey key)
        {
            IVersioned<T> val;
            objects.TryGetValue(key, out val);
            return val;
        }

        /// <summary>
        /// If the object with the specified identity had an
        /// existing instance at the specified time, return
        /// the versioned object. Otherwise return null.
        /// </summary>
        /// <param name="key">The identity of the object being sought.</param>
        /// <param name="when">The time for which we are inspecting it.</param>
        /// <returns>The versioned object requested, or if it did not
        /// exist at that time, a null reference.</returns>
        
        public IVersioned<T> FindIfExistedAt(IKey key, DateTime when)
        {
            IVersioned<T> vt = Find(key);
            if (vt != null && vt.ExistedAt(when))
                return vt;
            else
                return null;
        }

        /// <summary>
        /// Find the specific unversioned object with the specified identity as
        /// it was at the stipulated time.
        /// </summary>
        /// <param name="key">The identity of the object being sought.</param>
        /// <param name="when">The time for which we are inspecting it.</param>
        /// <returns>The unversioned object requested, or if it did not
        /// exist at that time, a default value for its type.</returns>
        
        public T FindAt(IKey key, DateTime when)
        {
            IVersioned<T> vt = FindIfExistedAt(key, when);
            if (vt != null)
                return vt.At(when);
            else
                return default(T);
        }
    }
}
