﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace VersionLib
{
    public interface IVersioned<T>
    {
        /// <summary>
        /// A user-allocated primary key or identity for
        /// this object, throughout its tracked versions
        /// </summary>

        IKey ID
        {
            get;
        }

        /// <summary>
        /// Was there an object in existence at the specified time and date?
        /// </summary>
        /// <param name="when">The time and date to test for</param>
        /// <returns>True if the object was in existence, false if not</returns>

        bool ExistedAt(DateTime when);

        /// <summary>
        /// Reference to value at the specified time and date. If there is a possibility
        /// that the item did not exist at the specified time, this will return
        /// null if the type is a class. If a value type, the caller should
        /// use ExistedAt before calling this method.
        /// </summary>
        /// <param name="when">Time at which we want the object</param>
        /// <returns>The object as it was at the specified time</returns>

        T At(DateTime when);

        /// <summary>
        /// Private copy of value at the specified time and date. If there is a possibility
        /// that the item did not exist at the specified time, this will return
        /// null if the type is a class. If a value type, the caller should
        /// use ExistedAt before calling this method.
        /// </summary>
        /// <param name="when">Time at which we want the object</param>
        /// <returns>Copy of the object as it was at the specified time</returns>

        T CopyAt(DateTime when);

        /// <summary>
        /// Obtain reference to the current value for the tracked object.
        /// Yields a null reference if the object currently has no value
        /// stored, or default(T) if a value type.
        /// </summary>

        T Value
        {
            get;
        }

        /// <summary>
        /// Obtain a copy of the current value for the tracked object.
        /// Yields/ a null reference if the object currently has no value
        /// stored, or default(T) if a value type.
        /// </summary>
        /// <returns>A clone of the current value of the versioned object</returns>

        T CopyValue();

        /// <summary>
        /// If desired, create a deep copy of the object. Insert it with a
        /// specified date and time stamp.
        /// </summary>
        /// <param name="when">The date and time to associate with the entry</param>
        /// <param name="value">The object to be associated with that time</param>
        /// <param name="copyObject">True to create a private copy of the
        /// object to be inserted, false to just use the reference</param>

        void InsertAt(DateTime when, T value, bool copyObject);
    }

    /// <summary>
    /// The historic listing for a particular object
    /// </summary>
    /// <typeparam name="T">The type of object this is a history of</typeparam>
    
    [Serializable]
    public class Versioned<T> : IVersioned<T>
    {
        /// <summary>
        /// The collection of historic values
        /// </summary>
        
        private List<IVersionOf<T>> valueList;

        /// <summary>
        /// Constructor. Allocates primary key
        /// to this object history, and creates
        /// the empty list of values
        /// <param name="kf">Factory to use for allocating unique keys</param>
        /// </summary>
        
        public Versioned(IKeyFactory kf)
        {
            ID = kf.NewKey();
            valueList = new List<IVersionOf<T>>();
        }

        /// <summary>
        /// A user-allocated primary key or identity for
        /// this object, throughout its tracked versions
        /// </summary>
        
        public IKey ID
        {
            get;
            private set;
        }

        /// <summary>
        /// Was there an object in existence at the specified time and date?
        /// </summary>
        /// <param name="when">The time and date to test for</param>
        /// <returns>True if the object was in existence, false if not</returns>
        
        public bool ExistedAt(DateTime when)
        {
            return valueList.Count > 0 && valueList.Last().Started < when;
        }

        /// <summary>
        /// Reference to value at the specified time and date. If there is a possibility
        /// that the item did not exist at the specified time, this will return
        /// null if the type is a class. If a value type, the caller should
        /// use ExistedAt before calling this method.
        /// </summary>
        /// <param name="when">Time at which we want the object</param>
        /// <returns>The object as it was at the specified time</returns>
        
        public T At(DateTime when)
        {
            // Create an empty IVersionOf<T> that just has the
            // date-time in it. This is used by BinarySearch to
            // do the successive comparisons needed to rapidly
            // find the selected version of the object.

            IVersionOf<T> epoch = VersionLibPackage.Container
                .GetInstance<IVersionOfFactory<T>>().Create(when);
            int location = valueList.BinarySearch(epoch);

            // If no item with exactly this timestamp existed,
            // the BinarySearch method returns the ones complement
            // of the index of the immediately preceding item. This
            // means the item whose values would have been valid
            // at the time selected.

            if (location < 0 && location != ~valueList.Count )
                return valueList[~location].Value;

            // If the returned location index is zero or positive,
            // this means that the exact date and time have been
            // found. In this case we return the item at that
            // exact position in the list.

            else if (location >= 0 && location < valueList.Count)
                return valueList[location].Value;
            else
                return default(T);
        }

        /// <summary>
        /// Private copy of value at the specified time and date. If there is a possibility
        /// that the item did not exist at the specified time, this will return
        /// null if the type is a class. If a value type, the caller should
        /// use ExistedAt before calling this method.
        /// </summary>
        /// <param name="when">Time at which we want the object</param>
        /// <returns>Copy of the object as it was at the specified time</returns>

        public T CopyAt(DateTime when)
        {
            return At(when).DeepClone();
        }

        /// <summary>
        /// Obtain reference to the current value for the tracked object.
        /// Yields a null reference if the object currently has no value
        /// stored, or default(T) if a value type.
        /// </summary>
        
        public T Value
        {
            get
            {
                if (valueList.Count == 0)
                    return default(T);
                else
                    return valueList.First().Value;
            }
        }

        /// <summary>
        /// Obtain a copy of the current value for the tracked object.
        /// Yields a null reference if the object currently has no value
        /// stored, or default(T) if a value type.
        /// </summary>
        /// <returns>A clone of the current value of the versioned object</returns>

        public T CopyValue()
        {
            return Value.DeepClone();
        }

        /// <summary>
        /// If desired, create a deep copy of the object. Insert it with a
        /// specified date and time stamp.
        /// </summary>
        /// <param name="value">The object to be associated with that time</param>
        /// <param name="when">The date and time to associate with the entry</param>
        /// <param name="copyObject">True to create a private copy of the
        /// object to be inserted, false to just use the reference</param>
        
        public void InsertAt(DateTime when, T value, bool copyObject)
        {
            // Create a new tracked object that contains a clone of the value so
            // that the original object can continue to have its data varied

            IVersionOfFactory<T> vf = VersionLibPackage.Container
                .GetInstance<IVersionOfFactory<T>>();
            IVersionOf<T> tt = vf.Create(when, value, copyObject);
            
            // Search the list of timestamps for the
            // index of the value that has the timestamp
            // passed in as an argument

            int location = valueList.BinarySearch(new VersionOf<T>(when));
            if (location < 0)
            {
                // When the location is negative, it means
                // that the exact match was not found. In
                // this case, the location value is the
                // bitwise ones-complement of the next
                // item in the list beyond the value
                // being sought. This is also the
                // correct index at which to do an
                // insertion.

                valueList.Insert(~location, tt);
            }
            else
            {
                // An exact match was found, meaning
                // that an item with this timestamp
                // was found in the collection. In
                // that case, overwrite the value.

                valueList[location] = tt;
            }
        }
    }
}
