﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace VersionLib
{
    public interface IVersionOf<T> : IComparable<IVersionOf<T>>
    {
        /// <summary>
        /// The time/date at which the object was timestamped
        /// </summary>

        DateTime Started
        {
            get;
        }

        /// <summary>
        /// The contained object with values set at specified time
        /// </summary>

        T Value
        {
            get;
        }
    }

    /// <summary>
    /// An instance of a T together with a timestamp at which
    /// it took on this set of values.
    /// </summary>
    /// <typeparam name="T">The type of object that is being timed</typeparam>
    
    [Serializable]
    public class VersionOf<T> : IVersionOf<T>
    {
        /// <summary>
        /// Constructor. Objects of this type
        /// are creatable, but their contents
        /// cannot subsequently be changed
        /// </summary>
        /// <param name="starts">Timestamp for the object's creation</param>
        /// <param name="val">The object that is being tracked</param>
        /// <param name="copyObject">True if a deep copy of the
        /// object passed as an argument should be made, false
        /// if we are happy to just insert the object reference.</param>
        
        public VersionOf(DateTime starts, T val, bool copyObject)
        {
            Started = starts;

            // For performance reasons, only do the
            // deep copy if the item is an object. Note
            // there may be a problem with the simplistic
            // code below if val is a value type but
            // contains embedded references.

            if (!copyObject || val is ValueType || val is string)
                Value = val;
            else
                Value = val.DeepClone();
        }

        /// <summary>
        /// Used so that BinarySearch can search a list of tracked objects.
        /// Note that the Value property is never initialised, as it is
        /// never used. We are only interested in the time for a BinarySearch.
        /// </summary>
        /// <param name="starts">Time to search for</param>
        
        public VersionOf(DateTime starts)
        {
            Started = starts;
        }

        /// <summary>
        /// The time/date at which the object was timestamped
        /// </summary>
        
        public DateTime Started
        {
            get;
            private set;
        }

        /// <summary>
        /// The contained object
        /// </summary>
        
        public T Value
        {
            get;
            private set;
        }

        /// <summary>
        /// Default sort order is in decreasing order of recency,
        /// or in other words, the most recent item at the front of the list.
        /// </summary>
        /// <param name="other">The object against which we shall
        /// compare ourselves</param>
        /// <returns>-1 if this object should be earlier in the collation order,
        /// 0 if the items should be simultaneous, +1 if the current object
        /// should be later than the argument.</returns>
        
        public int CompareTo(IVersionOf<T> other)
        {
            return Math.Sign((other.Started - Started).Ticks);
        }
    }
}
