﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace VersionLib
{
    /// <summary>
    /// Interface to a class that manufactures primary keys
    /// </summary>
    
    public interface IKeyFactory
    {
        /// <summary>
        /// Register a list of existing keys that are to be
        /// excluded from the allocation mechanism, so that
        /// all future keys are unique.
        /// </summary>
        /// <param name="existingKeys">The list of existing
        /// keys that are already in use</param>
        
        void RegisterExistingKeys(IEnumerable<IKey> existingKeys);

        /// <summary>
        /// Allocate a new, unused unique key
        /// </summary>
        /// <returns>The next free key in the key sequence</returns>
        
        IKey NewKey();
    }

    public class KeyFactory : IKeyFactory
    {
        private IKey maxKey = null;

        /// <summary>
        /// Register a list of existing keys that are to be
        /// excluded from the allocation mechanism, so that
        /// all future keys are unique.
        /// </summary>
        /// <param name="existingKeys">The list of existing
        /// keys that are already in use</param>

        public void RegisterExistingKeys(IEnumerable<IKey> existingKeys)
        {
            if(existingKeys != null && existingKeys.Any())
            {
                IKey newMaxKey = existingKeys.Max();
                if (maxKey == null || maxKey.CompareTo(newMaxKey) < 0)
                    maxKey = newMaxKey;
            }
        }

        /// <summary>
        /// Allocate a new, unused unique key
        /// </summary>
        /// <returns>The next free key in the key sequence</returns>

        public IKey NewKey()
        {
            if (maxKey == null)
                maxKey = new Key(1);
            else
                maxKey = maxKey.FollowingKey();
            return maxKey;
        }
    }
}
