﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace VersionLib
{
    /// <summary>
    /// The IKey&lt;T&gt; interface is used to describe any data type that
    /// is both comparable and equatable with another object of its own type.
    /// </summary>
    
    public interface IKey :
        IComparable<IKey>, 
        IEquatable<IKey> 
    {
        /// <summary>
        /// The next key beyond the current key, returned as a new key
        /// </summary>
        /// <returns>A new key with value incrementally greater
        /// than the current key</returns>
        
        IKey FollowingKey();
    }

    public interface IKeyWithValue<T> : IKey
    {
        /// <summary>
        /// The stored value for the unique key
        /// </summary>

        T Value
        {
            get;
        }
    }

    /// <summary>
    /// A basic integer implementation of the key type
    /// </summary>
    /// <typeparam name="T">The type used for the actual key value. Typically some kind of
    /// an integer, a string, or even a GUID.</typeparam>
    
    [Serializable]
    public struct Key : IKeyWithValue<int>
    {
        /// <summary>
        /// The stored value for the unique key
        /// </summary>

        public int Value
        {
            get;
            private set;
        }

        /// <summary>
        /// Constructor. The stored value in the key can be
        /// setup on construction, but thereafter is immutable.
        /// </summary>
        /// <param name="newValue">The new value to assign to the key</param>
        
        public Key(int newValue) : this()
        {
            Value = newValue;
        }

        /// <summary>
        /// Compare this key to another for ordering. Note that the
        /// implementation merely delegates to the T type, that is
        /// required to implement the same IComparable interface.
        /// </summary>
        /// <param name="other">The other key this key is being
        /// compared to.</param>
        /// <returns>-1, 0, or +1</returns>
        
        public int CompareTo(IKey other)
        {
            Key otherKey = (Key)other;
            return Value.CompareTo(otherKey.Value);
        }

        /// <summary>
        /// Compare this key to another for equality. Note that the
        /// implementation merely delegates to the T type, that is
        /// required to implement the same IEquatable interface.
        /// </summary>
        /// <param name="other">The other key this key is being
        /// compared to.</param>
        /// <returns>True if deemed equal</returns>

        public bool Equals(IKey other)
        {
            Key otherKey = (Key)other;
            return Value.Equals(otherKey.Value);
        }

        /// <summary>
        /// Return a new key with the value
        /// incrementally beyond the current key
        /// </summary>
        /// <returns>The next key beyond the current key</returns>
        
        public IKey FollowingKey()
        {
            return new Key(Value + 1);
        }
    }
}
