﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace VersionLib
{
    /// <summary>
    /// A unique key for a user story. Held in the format
    /// number|string (. number|string)+ 
    /// </summary>
    
    public class HierarchicalKey : IKeyWithValue<string>
    {
        /// <summary>
        /// The actual fully normalised key value
        /// </summary>
        
        public string Value
        {
            get;
            private set;
        }

        /// <summary>
        /// The prefix to the key. This is all characters
        /// up to but excluding the last period. If there are
        /// no periods in the key, the prefix is empty.
        /// </summary>
        
        public string Prefix
        {
            get
            {
                return GetPrefix(Value);
            }
            set
            {
                SetKey(value, Suffix);
            }
        }

        /// <summary>
        /// The suffix to the key. This is all characters
        /// after the last period in the key. If there is
        /// no period in the key, the suffix is the same
        /// as the key.
        /// </summary>
        
        public string Suffix
        {
            get
            {
                return GetSuffix(Value);
            }
            set
            {
                SetKey(Prefix, value);
            }
        }

        /// <summary>
        /// Apply correct formatting rules to set up the key string
        /// </summary>
        /// <param name="prefix">Characters up to but excluding the last period</param>
        /// <param name="suffix">Characters after the last period</param>
        
        private void SetKey(string prefix, string suffix)
        {
            if (string.IsNullOrEmpty(prefix))
                Value = suffix;
            else if (string.IsNullOrEmpty(suffix))
                Value = prefix;
            else
                Value = prefix + '.' + suffix;
        }

        /// <summary>
        /// Find the prefix to a key. This is all characters
        /// up to but excluding the last period. If there are
        /// no periods in the key, the prefix is empty.
        /// </summary>
        /// <param name="key">The key from which we wish to
        /// extract the prefix</param>
        /// <returns>The extracted prefix, if it exists</returns>
        
        private static string GetPrefix(string key)
        {
            if (string.IsNullOrEmpty(key))
                return string.Empty;

            int lastDotPos = key.LastIndexOf('.');
            if (lastDotPos <= 0)
                return string.Empty;
            else
                return key.Substring(0, lastDotPos);
        }

        /// <summary>
        /// Get the suffix to a key. This is all characters
        /// after the last period in the key. If there is
        /// no period in the key, the suffix is the same
        /// as the key.
        /// </summary>
        /// <param name="key">The key from which to extract the suffix</param>
        /// <returns>The suffix to the key</returns>
        
        private static string GetSuffix(string key)
        {
            if (string.IsNullOrEmpty(key))
                return string.Empty;

            int lastDotPos = key.LastIndexOf('.');
            return key.Substring(lastDotPos + 1);
        }

        /// <summary>
        /// Create the key by value
        /// </summary>
        /// <param name="keyStr">The string to use as the key value</param>
        
        public HierarchicalKey(string keyStr)
        {
            if (string.IsNullOrEmpty(keyStr))
                throw new ArgumentException("Key cannot be empty");
            else if (keyStr.StartsWith(".") || keyStr.EndsWith(".") || keyStr.IndexOf("..") > 0)
                throw new ArgumentException("Key must be identifiers or numbers separated by periods");
            else
                Value = NormaliseKey(keyStr);
        }

        /// <summary>
        /// Create the key from an integer value
        /// </summary>
        /// <param name="val">The value to be converted to a string key</param>

        public HierarchicalKey(int val)
            : this(val.ToString())
        {
        }

        /// <summary>
        /// Convert the key to a standard format that
        /// makes lexicographic comparison also numeric
        /// for the embedded digit strings
        /// </summary>
        /// <param name="k">The denormalised key</param>
        /// <returns>A normalised key</returns>
        
        private static string NormaliseKey(string k)
        {
            StringBuilder sb = new StringBuilder();
            string[] segs = k.Split('.');
            foreach(string seg in segs)
            {
                int v;
                if (int.TryParse(seg, out v))
                    sb.AppendFormat("{0:0000}.", v);
                else
                    sb.Append(seg + ".");
            }
            if(sb.Length > 0)
                sb.Remove(sb.Length - 1, 1);
            return sb.ToString();
        }

        /// <summary>
        /// Convert the key to a shorter format that
        /// is more readable by humans
        /// </summary>
        /// <param name="k">The normalised key</param>
        /// <returns>A denormalised key</returns>

        private static string DenormaliseKey(string k)
        {
            StringBuilder sb = new StringBuilder();
            string[] segs = k.Split('.');
            foreach (string seg in segs)
            {
                int v;
                if (int.TryParse(seg, out v))
                    sb.Append(v.ToString() + ".");
                else
                    sb.Append(seg + ".");
            }
            if (sb.Length > 0)
                sb.Remove(sb.Length - 1, 1);
            return sb.ToString();
        }

        /// <summary>
        /// Obtain the key value
        /// </summary>
        /// <returns>The key value as a string</returns>
        
        public override string ToString()
        {
            return DenormaliseKey(Value);
        }

        /// <summary>
        /// Compare this key with another
        /// </summary>
        /// <param name="obj">The other key to compare with</param>
        /// <returns>True if the same, false if not</returns>
        
        public bool Equals(IKey obj)
        {
            IKeyWithValue<string> other = obj as IKeyWithValue<string>;
            return other != null && string.Compare(Value, other.Value, true) == 0;
        }

        /// <summary>
        /// Included to stop the compiler commplaining
        /// </summary>
        /// <returns>A unique hash code for the object</returns>
        
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }

        /// <summary>
        /// Implement normal lexicographic sort order for keys
        /// </summary>
        /// <param name="other">The other key against which we are comparing</param>
        /// <returns>Standard IComparable return value</returns>

        public int CompareTo(IKey other)
        {
            return string.Compare(Value, ((IKeyWithValue<string>)other).Value, true);
        }

        /// <summary>
        /// Return true if the leading characters in the
        /// key match in a case-insensitive comparison
        /// </summary>
        /// <param name="kh">The leading characters to compare</param>
        /// <returns>True if they match, false if not</returns>
        
        public bool LeadingElementsMatch(string kh)
        {
            return LeadingElementsMatch(Value, kh);
        }

        /// <summary>
        /// Return true if the leading characters of the first string
        /// parameter match the characters in the second parameter. The
        /// comparison is case-insensitive. The match is done on whole
        /// blocks delimited by periods.
        /// </summary>
        /// <param name="k">The string being tested to see if it begins
        /// with the characters in the second argument</param>
        /// <param name="kh">The pattern being sought at the beginning
        /// of the first string parameter</param>
        /// <returns>True if the pattern is found</returns>
        
        public static bool LeadingElementsMatch(string k, string kh)
        {
            // Normalise the comparison string

            if (string.IsNullOrEmpty(kh))
                kh = string.Empty;
            else if (!kh.EndsWith("."))
                kh += ".";

            // Make sure the leading elements take us down to the
            // last segment in the key value.

            if (k.LastIndexOf('.') != kh.Length - 1)
                return false;
            else
                return string.Compare(kh, 0, k, 0, kh.Length, true) == 0;
        }

        /// <summary>
        /// Find a new normalised next key value for this key
        /// </summary>
        /// <returns>A new key, or null if the last part of
        /// the key was non-numeric, or if the argument
        /// was null or empty</returns>
        
        public static HierarchicalKey NextKey(string prevKey)
        {
            // Find the two segments of the key around the last '.',
            // so that "Fred.0031" becomes "Fred" and "0031"

            string prefix = GetPrefix(prevKey);
            string suffix = GetSuffix(prevKey);

            int v = 0;
            if (!int.TryParse(suffix, out v))
                return null;

            // Construct the next string in the sequence. This is
            // not very efficient, as it applies key normalisation again.

            return new HierarchicalKey(prefix + "." + (v + 1).ToString());
        }

        /// <summary>
        /// Obtain the last part of the key as an integer. If the
        /// last part of the key does not consist solely of digits,
        /// return the value 0.
        /// </summary>
        
        public int LastNumericSegment
        {
            get
            {
                if (!string.IsNullOrEmpty(Value))
                {
                    string suffix = GetSuffix(Value);
                    int v = 0;
                    if (int.TryParse(suffix, out v))
                        return v;
                }
                return 0;
            }
        }

        /// <summary>
        /// The next valued key beyond this one
        /// </summary>
        /// <returns>A key that is considered an increment
        /// beyond the current key</returns>
        
        public IKey FollowingKey()
        {
            return NextKey(Value);
        }
    }
}
