﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Web.Http;
using TFSLib;
using System.Web.Http.Cors;
using System.Drawing;
using System.IO;
using System.Net.Http.Headers;
using System.Text.RegularExpressions;

namespace TFSTools.Controllers
{
    [EnableCors(origins: "*", headers: "*", methods: "*")]
    public class ViolationController : ApiController
    {
        private static Dictionary<string, List<ViolationSet>> violations = null;

        /// <summary>
        /// Retrieve the list of all rule violations
        /// </summary>

        public static List<ViolationSet> Violations(string project)
        {
            if(violations == null)
                violations = new Dictionary<string,List<ViolationSet>>();
            if(!violations.ContainsKey(project))
                violations.Add(project, new List<ViolationSet>(0));
            if (violations[project] == null)
                RefreshViolations(project);
            return violations[project];
        }

        /// <summary>
        /// Insert or overwrite a set of violations for a project
        /// </summary>
        /// <param name="project">The TFS project name, e.g. DataWarehouse or FTP</param>
        /// <param name="viols">The set of rule violations for that project</param>
        
        public static void SetViolations(string project, List<ViolationSet> viols)
        {
            if(violations == null)
                violations = new Dictionary<string,List<ViolationSet>>();
            violations[project] = viols;
        }

        /// <summary>
        /// Bring the set of violations up to date by forcing
        /// a complete reload from the TFS databases
        /// </summary>

        private static void RefreshViolations(string project)
        {
            SetViolations(project, RulesEngine.Run
                (WorkItemAdapter.ItemAdapter(project)));
        }

        /// <summary>
        /// General JPG image downloader
        /// </summary>
        /// <param name="plotData">String descriptor used to determine 
        /// the JPG stream to generate</param>
        /// <returns>The JPG image stream requested</returns>
        
        [Route("api/plot/{plotData}")]
        [HttpGet]
        public HttpResponseMessage Get(string plotData)
        {
            Image img = GenerateImage(plotData);
            MemoryStream ms = new MemoryStream();
            img.Save(ms, System.Drawing.Imaging.ImageFormat.Jpeg);
            img.Dispose();
            HttpResponseMessage result = new HttpResponseMessage(HttpStatusCode.OK);
            result.Content = new ByteArrayContent(ms.ToArray());
            result.Content.Headers.ContentType = new MediaTypeHeaderValue("image/jpg");
            return result;
        } 

        private Image GenerateImage(string plotData)
        {
            Bitmap bmp = new Bitmap(400, 400);
            Graphics g = Graphics.FromImage(bmp);
            Pen p = new Pen(Color.Red, 12.0F);
            g.DrawEllipse(p, 6, 6, 388, 388);
            g.Dispose();
            return bmp;
        }
        
        /// <summary>
        /// Look up the set of all Items from the work item store.
        /// This is hard-wired for the DataWarehouse project.
        /// </summary>
        /// <returns>The set of all items from the work item store</returns>

        [Route("api/Violation")]
        [HttpGet]
        public IEnumerable<ViolationSet> Get()
        {
            return GetByProject("DataWarehouse");
        }

        /// <summary>
        /// Look up the set of all Items from the work item store.
        /// Should not be used as it returns a large number of items.
        /// </summary>
        /// <param name="project">The name of the TFS project to search</param>
        /// <returns>The set of all items from the work item store</returns>

        [Route("api/Violation/{project}")]
        [HttpGet]
        public IEnumerable<ViolationSet> GetByProject(string project)
        {
            WorkItemAdapter wia = WorkItemAdapter.ItemAdapter(project);
            if (wia != null)
            {
                RefreshViolations(project);
                return Violations(project);
            }
            else
                return Enumerable.Empty<ViolationSet>();
        }

        /// <summary>
        /// Find an item by its specific primary key value. For the
        /// work item store in TFS this means the System.Id value for
        /// a work item.
        /// </summary>
        /// <param name="id">The unique integer ID of the work item, as previously
        /// allocated by TFS</param>
        /// <returns>The found item as a collection of one item, or no items if not found</returns>

        [Route("api/Violation/{id:int}")]
        [HttpGet]
        public IEnumerable<ViolationSet> Get(int id)
        {
            return GetByProject("DataWarehouse", id);
        }

        /// <summary>
        /// Find an item by its specific primary key value. For the
        /// work item store in TFS this means the System.Id value for
        /// a work item.
        /// </summary>
        /// <param name="project">The TFS project name</param>
        /// <param name="id">The unique integer ID of the work item, as previously
        /// allocated by TFS</param>
        /// <returns>The found item as a collection of one item, or no items if not found</returns>

        [Route("api/Violation/{project}/{id:int}")]
        [HttpGet]
        public IEnumerable<ViolationSet> GetByProject(string project, int id)
        {
            WorkItemAdapter wia = WorkItemAdapter.ItemAdapter(project);
            if (wia != null)
            {
                List<ViolationSet> viols = Violations(project);
                if (viols.Count == 0)
                    RefreshViolations(project);
                return Violations(project).Where(v => v.Source.Id == id);
            }
            else
                return Enumerable.Empty<ViolationSet>();
        }

        /// <summary>
        /// Find all end of sprint items matching the sprint and area.
        /// </summary>
        /// <param name="esFilter">Contains filters used to limit returned items</param>
        /// <returns>The found items, or no items if no matches</returns>

        [Route("api/SprintItems")]
        [HttpPost]
        public IEnumerable<EndSprintItem> Post([FromBody]EndSprintFilter esFilter)
        {
            if (esFilter.Area == null)
                esFilter.Area = "DataWarehouse\\Stream 1";
            if (esFilter.Iteration == null)
                esFilter.Iteration = "DataWarehouse\\Sprint 63";

            string project = TFSProjectNameFromPath(esFilter.Area);
            if(project != null && project == TFSProjectNameFromPath(esFilter.Iteration))
            {
                WorkItemAdapter wia = WorkItemAdapter.ItemAdapter(project);
                if(wia != null)
                {
                    SprintManager sm = new SprintManager(wia, esFilter.Area, esFilter.Iteration);
                    return sm.AllSprintItems;
                }
            }
            return Enumerable.Empty<EndSprintItem>();
        }

        private string TFSProjectNameFromPath(string path)
        {
            int sepIdx = path.IndexOf('\\');
            if (sepIdx > 0)
                return path.Substring(0, sepIdx);
            return null;
        }

        /// <summary>
        /// For the nominated work item, roll it to the next sprint 
        /// or back onto the product backlog, dependig on the value
        /// of the second argument.
        /// </summary>
        /// <param name="id">Id of the TFS work item to be rolled</param>
        /// <param name="idx">0 for roll over, 1 for putting back
        /// onto product backlog</param>
        /// <returns>Empty string if successful, error message if failed</returns>

        [Route("api/Rollover/{id}/{idx}")]
        [HttpGet]
        public string RollOver(int id, int idx)
        {
            return RollOver("DataWarehouse", id, idx);
        }

        /// <summary>
        /// Given the sprint number, compute the tfs usage
        /// statistics for each TFS team member, grouped 
        /// into their teams
        /// </summary>
        /// <param name="sprint">The sprint for which we 
        /// desire the statistics</param>
        /// <returns>The per user statistics for the
        /// selected sprint</returns>
        
        [Route("api/TfsUsage")]
        [HttpPost]
        public List<TFSUsageGroup> PostUsageReport([FromBody]EndSprintFilter sprint)
        {
            if (sprint != null && !string.IsNullOrEmpty(sprint.Iteration))
            {
                ReportManager rm = new ReportManager("DataWarehouse");
                return rm.TfsUsageReport(sprint.Iteration);
            }
            else
                return new List<TFSUsageGroup>(0);
        }

        /// <summary>
        /// Generate velocity and estimates statistics
        /// for scrum teams ad iterations
        /// </summary>
        /// <returns>The list of statistics</returns>
        
        [Route("api/Stats")]
        [HttpPost]
        public List<TeamSprintStats> PostStatsReport([FromBody]EndSprintFilter sprint)
        {
            if (sprint != null)
            {
                ReportManager rm = new ReportManager("DataWarehouse");
                return rm.CalcStats(sprint.Area, sprint.Iteration);
            }
            else
                return new List<TeamSprintStats>(0);
        }

        /// <summary>
        /// For the nominated work item, roll it to the next sprint 
        /// or back onto the product backlog, dependig on the value
        /// of the second argument.
        /// </summary>
        /// <param name="project">The TFS project to apply to</param>
        /// <param name="id">Id of the TFS work item to be rolled</param>
        /// <param name="idx">0 for roll over, 1 for putting back
        /// onto product backlog</param>
        /// <returns>Empty string if successful, error message if failed</returns>

        [Route("api/Rollover/{project}/{id}/{idx}")]
        [HttpGet]
        public string RollOver(string project, int id, int idx)
        {
            WorkItemAdapter wia = WorkItemAdapter.ItemAdapter(project);
            if(wia != null)
            {
                var foundItem = wia.Select(i => i.Id == id).FirstOrDefault();
                if(foundItem == null)
                    return "No work item with specified ID in this TFS project";
                try
                {
                    SprintManager sm = new SprintManager
                        (wia, foundItem.AreaPath, foundItem.IterationPath);
                    sm.RollOver(foundItem, idx == 0, null);
                    return string.Empty;
                }
                catch(Exception x)
                {
                    return x.Message;
                }
            }
            return "No TFS project named " + project;
        }

        /// <summary>
        /// Find all items matching the search criteria.
        /// </summary>
        /// <param name="vFilter">Contains filters used to limit returned items</param>
        /// <returns>The found items, or no items if no matches</returns>

        [Route("api/Violation")]
        [HttpPost]
        public IEnumerable<ViolationSet> Post([FromBody]ViolationFilter vFilter)
        {
            WorkItemAdapter wia;
            if (!string.IsNullOrEmpty(vFilter.Project))
                wia = WorkItemAdapter.ItemAdapter(vFilter.Project);
            else
                wia = WorkItemAdapter.ItemAdapter("DataWarehouse");

            // Negative day count forces TFS refresh (can take several tens of seconds)

            if (wia != null)
            {
                if (vFilter.Days < 0 || Violations(vFilter.Project).Count == 0)
                    RefreshViolations(wia.TfsProjectName);
                IEnumerable<ViolationSet> query = Violations(vFilter.Project).AsEnumerable();
                if (vFilter.Days > 0)
                    query = query.Where(v => v.Source.ChangedDate > DateTime.Today.AddDays(-vFilter.Days));
                if (vFilter.Area != null && vFilter.Area != "-")
                    query = query.Where(v => v.Source.AreaPath
                        .StartsWith(vFilter.Area, StringComparison.CurrentCultureIgnoreCase));
                if (vFilter.Iteration != null && vFilter.Iteration != "-")
                    query = query.Where(v => v.Source.IterationPath
                        .StartsWith(vFilter.Iteration, StringComparison.CurrentCultureIgnoreCase));
                if (vFilter.Editor != null && vFilter.Editor != "-")
                    query = query.Where(v => v.Source.ChangedBy
                        .StartsWith(vFilter.Editor, StringComparison.CurrentCultureIgnoreCase));
                return query;
            }
            else
                return Enumerable.Empty<ViolationSet>();
        }

        /// <summary>
        /// Force a reload of all the violations
        /// </summary>
        /// <returns>True always</returns>

        [Route("api/Reset")]
        [HttpGet]
        public bool GetNew()
        {
            return GetNew("DataWarehouse");
        }

        /// <summary>
        /// Force a reload of all the violations
        /// </summary>
        /// <returns>True always</returns>

        [Route("api/Reset/{project}")]
        [HttpGet]
        public bool GetNew(string project)
        {
            WorkItemAdapter.RefreshAdapter(project);
            SetViolations(project, null);
            return true;
        }

        /// <summary>
        /// Obtain the data needed to apply filters to lists of violations.
        /// </summary>
        /// <returns>The list of all areas (streams), all sprints
        /// (iterations), and all last change authors</returns>

        [Route("api/Filters")]
        [HttpGet]
        public ViolationFilters GetFilters()
        {
            return GetFilters("DataWarehoue");
        }

        /// <summary>
        /// Obtain the data needed to apply filters to lists of violations.
        /// </summary>
        /// <returns>The list of all areas (streams), all sprints
        /// (iterations), and all last change authors</returns>

        [Route("api/Filters/{project}")]
        [HttpGet]
        public ViolationFilters GetFilters(string project)
        {
            WorkItemAdapter wia = WorkItemAdapter.ItemAdapter(project);
            if (wia != null)
            {
                return new ViolationFilters
                {
                    Areas = wia.AllCurrentWorkItems
                        .Select(i => i.AreaPath)
                        .Where(ap => ValidFilteredArea(ap))
                        .Distinct()
                        .OrderBy(ap => ap)
                        .ToArray(),
                    Iterations = wia.AllCurrentWorkItems
                        .Select(i => i.IterationPath)
                        .Where(ip => ValidFilteredIteration(ip))
                        .Distinct()
                        .OrderBy(ip => ip)
                        .ToArray(),
                    Editors = wia.AllCurrentWorkItems
                        .Select(i => i.ChangedBy)
                        .Distinct()
                        .OrderBy(ip => ip)
                        .ToArray()
                };
            }
            else
                return new ViolationFilters
                {
                    Areas = new string[0],
                    Iterations = new string[0],
                    Editors = new string[0]
                };
        }

        private Regex areaRE = new Regex("^.*\\Stream [a-zA-Z0-9\\.]+$");
        private bool ValidFilteredArea(string area)
        {
            return !area.Contains("x_Old") && areaRE.IsMatch(area);
        }

        private Regex iterRE = new Regex("^.*\\Sprint [a-zA-Z0-9\\.]+$");
        private bool ValidFilteredIteration(string iter)
        {
            return !iter.Contains("x_Old") && iterRE.IsMatch(iter);
        }
    }
}
