﻿/*
 * Author: 
 *  S D Smith
 *  
 * Date:
 *  Oct 2015
 *  
 * Purpose: 
 *  Unit tests for TFS rules checker library.
 *  Used as part of the TFS sanity checks to make sure TFS users in Analytics
 *  conform to recommended best practices in using the work items to
 *  plan and track backlogs and sprints.
 */

using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using TFSLib;
using System.Linq;
using System.Reflection;
using System.Collections.Generic;

namespace TFSLibTest
{
    [TestClass]
    public class WorkItemTests
    {
        private static WorkItemAdapter wiAdapter = null;

        private static WorkItemAdapter WIA
        {
            get
            {
                if (wiAdapter == null)
                    wiAdapter = WorkItemAdapter.ItemAdapter("DataWarehouse");
                return wiAdapter;
            }
        }
        [TestMethod]
        public void TestConnection()
        {
            Assert.IsNotNull(WIA);
        }

        [TestMethod]
        public void TestTfsServerNameGet()
        {
            Assert.AreEqual("http://tfs:8080/tfs/FTPDev/DataWarehouse", WIA.TfsProject);
        }

        [TestMethod]
        public void GetWorkItems()
        {
            var workItems = WIA.FindWorkItemsInTeamSprint
                ("Product Backlog Item", "DataWarehouse\\Sprint 62", "DataWarehouse\\Stream 1");
            Assert.AreEqual(16, workItems.Count());
        }

        [TestMethod]
        public void InsertWorkItem()
        {
            Item newItem = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is a demo of a PBI insertion",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                RemainingWork = 0.5,
                State = "New",
                Title = "Test of inserted PBI",
                WorkItemType = "Product Backlog Item"
            };
            string err = WIA.InsertItem(newItem);
            Assert.AreEqual(string.Empty, err);
            Assert.AreNotEqual(0, newItem.Id);
            Assert.IsFalse(newItem.Dirty);
        }

        [TestMethod]
        public void UpdateWorkItem()
        {
            Item newItem = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is a demo of a PBI insertion",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                RemainingWork = 0.5,
                State = "New",
                Title = "Test of inserted PBI",
                WorkItemType = "Product Backlog Item"
            };
            string err = WIA.InsertItem(newItem);
            Assert.AreEqual(string.Empty, err);
            newItem.Description = "This is an altered Description";
            Assert.IsTrue(newItem.Dirty);
            err = WIA.UpdateItem(newItem);
            Assert.AreEqual(string.Empty, err);
            Item changedItem = WIA.FindItem(newItem.Id);
            Assert.IsNotNull(changedItem);
            Assert.AreEqual(changedItem.Id, newItem.Id);
            Assert.AreEqual("This is an altered Description", changedItem.Description);
        }

        [TestMethod]
        public void GetIterationSchedule()
        {
            ScheduleInfo si = WIA.GetSprintSchedule("DataWarehouse\\Sprint 63");
            Assert.IsNotNull(si);
            Assert.AreEqual("DataWarehouse\\Sprint 63", si.IterationPath);
            Assert.AreEqual(new DateTime(2015, 10, 22), si.StartDate);
            Assert.AreEqual(new DateTime(2015, 11, 18), si.EndDate);
        }

        [TestMethod]
        public void ItemHistory()
        {
            Item item = WIA.FindItem(66249);
            Assert.IsNotNull(item);
            List<Item> history = WIA.ItemHistory(item);
            Assert.IsNotNull(history);
            Assert.AreEqual(9, history.Count);
        }

        [TestMethod]
        public void Teams()
        {
            Dictionary<string, List<string>> users = WIA.Teams;
            Assert.IsNotNull(users);
            Assert.IsTrue(users.Count > 0);
        }

        [TestMethod]
        public void TfsUsageReport()
        {
            ReportManager rm = new ReportManager("DataWarehouse");
            var usageReport = rm.TfsUsageReport("Sprint 64");
            Assert.IsNotNull(usageReport);
            Assert.IsTrue(usageReport.Count > 0);
        }

        [TestMethod]
        public void TfsCalcStats()
        {
            ReportManager rm = new ReportManager("DataWarehouse");
            var statsReport = rm.CalcStats("Stream 1", "Sprint 64");
            var otherStatsReport = rm.CalcStats("Stream 3", "Sprint 65");
            Assert.IsNotNull(statsReport);
            Assert.IsNotNull(otherStatsReport);
            Assert.IsTrue(statsReport.Count > 0);
        }

        [TestMethod]
        public void DeleteItem()
        {
            Item newItem = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is a demo of a PBI insertion",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                RemainingWork = 0.5,
                State = "New",
                Title = "Test of inserted PBI",
                WorkItemType = "Product Backlog Item"
            };
            string err = WIA.InsertItem(newItem);
            Assert.AreEqual(string.Empty, err);
            IEnumerable<Item> testItems = WIA.AllCurrentWorkItems
                .Where(wi => wi.Title == "Test of inserted PBI" 
                    || wi.Title == "Test of inserted Task");
            Assert.IsTrue(testItems.Any());
            foreach (Item i in testItems.ToList())
            {
                err = WIA.DeleteItem(i, false);
                Assert.IsTrue(string.IsNullOrEmpty(err));
            }
            testItems = WIA.AllCurrentWorkItems
                .Where(wi => wi.Title == "Test of inserted PBI"
                    || wi.Title == "Test of inserted Task");
            Assert.IsFalse(testItems.Any());
        }

        [TestMethod]
        public void InsertLink()
        {
            Item newItem = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is a demo of a PBI insertion",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                RemainingWork = 0.5,
                State = "New",
                Title = "Test of inserted PBI",
                WorkItemType = "Product Backlog Item"
            };
            string err = WIA.InsertItem(newItem);
            Assert.AreEqual(string.Empty, err);
            Item newTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is a demo of a Task insertion",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                RemainingWork = 0.5,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(newTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(newItem, newTask, "Child");
            Assert.AreEqual(string.Empty, err);
            var insertedLinks = WIA.SelectLinksBetween(newItem, newTask);
            Assert.IsNotNull(insertedLinks);
            Assert.AreEqual(1, insertedLinks.Count);
            Assert.AreEqual(newItem.Id, insertedLinks[0].Source.Id);
            Assert.AreEqual(newTask.Id, insertedLinks[0].Target.Id);
            Assert.AreEqual("Child", insertedLinks[0].LinkType);
        }

        [TestMethod]
        public void DeleteLink()
        {
            Item newItem = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is a demo of a PBI insertion",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                RemainingWork = 0.5,
                State = "New",
                Title = "Test of inserted PBI",
                WorkItemType = "Product Backlog Item"
            };
            string err = WIA.InsertItem(newItem);
            Assert.AreEqual(string.Empty, err);
            Item newTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is a demo of a Task insertion",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                RemainingWork = 0.5,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(newTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(newItem, newTask, "Child");
            Assert.AreEqual(string.Empty, err);
            ItemLink link = new ItemLink(newItem, newTask, "Child");
            err = WIA.DeleteLink(link);
            var insertedLinks = WIA.SelectLinksBetween(newItem, newTask);
            Assert.AreEqual(0, insertedLinks.Count);
        }

        private void CleanRollOverItems()
        {
            var itemsToBeDeleted = WIA
                .AllCurrentWorkItems
                .Where(i => i.AreaPath == "DataWarehouse\\Stream 5"
                    && i.Title.Contains("Test of inserted ")).ToList();
            var linksToBeDeleted = itemsToBeDeleted
                .Where(i => i.WorkItemType != "Task")
                .Select(i => WIA.ItemLinksFromSourceId(i.Id));

            // Safety check in case code deletes entire work item store

            if (itemsToBeDeleted.Count > 32)
                throw new InvalidOperationException
                    ("Safety check: Deleting too many items");

            // Remove all links first

            foreach (List<ItemLink> lil in linksToBeDeleted.ToList())
                foreach (ItemLink il in new List<ItemLink>(lil))
                    WIA.DeleteLink(il);

            // Now remove the work items

            foreach (Item i in itemsToBeDeleted.ToList())
                WIA.DeleteItem(i, false);
        }

        [TestMethod]
        public void RollOverZeroHoursPBI()
        {
            CleanRollOverItems();

            // Create the parent PBI in the previous print

            Item oldItem = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is a demo of a PBI insertion",
                Effort = 12.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                State = "New",
                Title = "Test of inserted PBI",
                WorkItemType = "Product Backlog Item"
            };
            string err = WIA.InsertItem(oldItem);
            Assert.AreEqual(string.Empty, err);
            oldItem.State = "Approved";
            err = WIA.UpdateItem(oldItem);
            Assert.AreEqual(string.Empty, err);
            oldItem.State = "Committed";
            err = WIA.UpdateItem(oldItem);
            Assert.AreEqual(string.Empty, err);

            // Create a task that will be in the
            // 'In Progress' state on sprint roll over,
            // but that has burned all its hours

            Item inProgTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is the in progress task",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                RemainingWork = 0.0,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(inProgTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(oldItem, inProgTask, "Child");
            Assert.AreEqual(string.Empty, err);

            // Advance through states

            inProgTask.State = "In Progress";
            err = WIA.UpdateItem(inProgTask);
            Assert.AreEqual(string.Empty, err);

            // Create a task that will be in the
            // 'Done' state on sprint roll over

            Item doneTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is the done task",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                RemainingWork = 0.0,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(doneTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(oldItem, doneTask, "Child");
            Assert.AreEqual(string.Empty, err);
            doneTask.State = "In Progress";
            err = WIA.UpdateItem(doneTask);
            Assert.AreEqual(string.Empty, err);
            doneTask.State = "Done";
            err = WIA.UpdateItem(doneTask);
            Assert.AreEqual(string.Empty, err);

            // Now roll this story over

            SprintManager sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 63");
            sm.RollOver(oldItem, true, null);

            // Explore sprint 64 stream 5 for output items

            sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 64");

            // Assertions on parent PBI in next sprint

            Item tstOldItem = sm.PBIs.FirstOrDefault(pbi => pbi.Id == oldItem.Id);
            Assert.IsNull(tstOldItem);

            // Assertions on PBI in completed sprint

            sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 63");
            Item tstNewItem = sm.PBIs.FirstOrDefault
                (pbi => pbi.Title == "Test of inserted PBI");
            Assert.IsNotNull(tstNewItem);
            Assert.AreEqual(12.0, tstNewItem.Effort);
            Assert.IsTrue(tstNewItem.State == "Done");

            // Assertions on child tasks of PBI in completed sprint

            var newChildLinks = sm.ItemAdapter.ItemLinksFromSourceId(tstNewItem.Id);
            Assert.IsNotNull(newChildLinks);
            Assert.AreEqual(2, newChildLinks.Count);
            var doneLink = newChildLinks.FirstOrDefault
                (cl => cl.Target.Description == "This is the done task");
            Assert.IsNotNull(doneLink);
            Assert.AreEqual(0.0, doneLink.Target.RemainingWork);
            Assert.AreEqual(doneTask.Id, doneLink.Target.Id);
            var newInProgLink = newChildLinks.FirstOrDefault
                (cl => cl.Target.Description == "This is the in progress task");
            Assert.IsNotNull(newInProgLink);
            Assert.AreEqual("Done", newInProgLink.Target.State);
            Assert.AreEqual(2.0, newInProgLink.Target.Effort);
            Assert.AreEqual(0.0, newInProgLink.Target.RemainingWork);
            CleanRollOverItems();
        }

        [TestMethod]
        public void RollOverUnstartedPBI()
        {
            CleanRollOverItems();

            // Create the parent PBI in the previous print

            Item oldItem = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is a demo of a PBI insertion",
                Effort = 12.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                State = "New",
                Title = "Test of inserted PBI",
                WorkItemType = "Product Backlog Item"
            };
            string err = WIA.InsertItem(oldItem);
            Assert.AreEqual(string.Empty, err);
            oldItem.State = "Approved";
            err = WIA.UpdateItem(oldItem);
            Assert.AreEqual(string.Empty, err);
            oldItem.State = "Committed";
            err = WIA.UpdateItem(oldItem);
            Assert.AreEqual(string.Empty, err);

            // Create a task that will be in the
            // 'In Progress' state on sprint roll over,
            // but that has burned no hours

            Item inProgTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is the in progress task",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                RemainingWork = 2.0,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(inProgTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(oldItem, inProgTask, "Child");
            Assert.AreEqual(string.Empty, err);

            // Advance through states

            inProgTask.State = "In Progress";
            err = WIA.UpdateItem(inProgTask);
            Assert.AreEqual(string.Empty, err);

            // Create a task that will be in the
            // 'To Do' state on sprint roll over

            Item toDoTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is the unstarted task",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 63",
                RemainingWork = 2.0,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(toDoTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(oldItem, toDoTask, "Child");
            Assert.AreEqual(string.Empty, err);

            // Now roll this story over

            SprintManager sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 63");
            sm.RollOver(oldItem, true, null);

            // Explore sprint 63 stream 5 for output items

            sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 63");

            // Assertions on parent PBI in previous sprint

            Item tstOldItem = sm.PBIs.FirstOrDefault(pbi => pbi.Id == oldItem.Id);
            Assert.IsNull(tstOldItem);

            // Assertions on PBI in next sprint

            sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 64");
            Item tstNewItem = sm.PBIs.FirstOrDefault
                (pbi => pbi.Title == "Test of inserted PBI");
            Assert.IsNotNull(tstNewItem);
            Assert.AreEqual(12.0, tstNewItem.Effort);
            Assert.IsTrue(tstNewItem.State != "Done");

            // Assertions on child tasks of PBI in next sprint

            var newChildLinks = sm.ItemAdapter.ItemLinksFromSourceId(tstNewItem.Id);
            Assert.IsNotNull(newChildLinks);
            Assert.AreEqual(2, newChildLinks.Count);
            var toDoLink = newChildLinks.FirstOrDefault
                (cl => cl.Target.Description == "This is the unstarted task");
            Assert.IsNotNull(toDoLink);
            Assert.AreEqual(2.0, toDoLink.Target.RemainingWork);
            Assert.AreEqual(toDoTask.Id, toDoLink.Target.Id);
            var newInProgLink = newChildLinks.FirstOrDefault
                (cl => cl.Target.Description == "This is the in progress task");
            Assert.IsNotNull(newInProgLink);
            Assert.AreEqual("To Do", newInProgLink.Target.State);
            Assert.AreEqual(2.0, newInProgLink.Target.Effort);
            Assert.AreEqual(2.0, newInProgLink.Target.RemainingWork);
            CleanRollOverItems();
        }

        [TestMethod]
        public void RollOverPartCompletePBI()
        {
            CleanRollOverItems();

            // Create the parent PBI in the previous print

            Item oldItem = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is a demo of a PBI insertion",
                Effort = 12.0,
                IterationPath = "DataWarehouse\\Sprint 64",
                State = "New",
                Title = "Test of inserted PBI",
                WorkItemType = "Product Backlog Item"
            };
            string err = WIA.InsertItem(oldItem);
            Assert.AreEqual(string.Empty, err);
            oldItem.State = "Approved";
            err = WIA.UpdateItem(oldItem);
            Assert.AreEqual(string.Empty, err);
            oldItem.State = "Committed";
            err = WIA.UpdateItem(oldItem);
            Assert.AreEqual(string.Empty, err);

            // Create a task that will be in the 'Done' state
            // on sprint roll over

            Item doneTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is the done task",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 64",
                RemainingWork = 2.0,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(doneTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(oldItem, doneTask, "Child");
            Assert.AreEqual(string.Empty, err);

            // Advance through states

            doneTask.State = "In Progress";
            err = WIA.UpdateItem(doneTask);
            Assert.AreEqual(string.Empty, err);
            doneTask.RemainingWork = 0;
            doneTask.State = "Done";
            err = WIA.UpdateItem(doneTask);
            Assert.AreEqual(string.Empty, err);

            // Create a task that will be in the
            // 'In Progress' state on sprint roll over

            Item inProgTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is the in progress task",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 64",
                RemainingWork = 2.0,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(inProgTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(oldItem, inProgTask, "Child");
            Assert.AreEqual(string.Empty, err);

            // Advance through states

            inProgTask.State = "In Progress";
            inProgTask.RemainingWork = 1.5;
            err = WIA.UpdateItem(inProgTask);
            Assert.AreEqual(string.Empty, err);

            // Create a task that will be in the
            // 'To Do' state on sprint roll over

            Item toDoTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is the unstarted task",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 64",
                RemainingWork = 2.0,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(toDoTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(oldItem, toDoTask, "Child");
            Assert.AreEqual(string.Empty, err);

            // Now roll this story over

            SprintManager sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 64");
            sm.RollOver(oldItem, true, null);

            // Explore sprint 63 stream 5 for output items

            sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 64");

            // Assertions on parent PBI in previous sprint

            Item tstOldItem = sm.PBIs.FirstOrDefault(pbi => pbi.Id == oldItem.Id);
            Assert.IsNotNull(tstOldItem);
            Assert.AreEqual(5.0, tstOldItem.Effort);
            Assert.AreEqual("Done", tstOldItem.State);

            // Assertions on tasks in previous sprint

            var childLinks = sm.ItemAdapter.ItemLinksFromSourceId(tstOldItem.Id);
            Assert.IsNotNull(childLinks);
            Assert.AreEqual(2, childLinks.Count);
            var doneLink = childLinks.FirstOrDefault(cl => cl.Target.Id == doneTask.Id);
            Assert.IsNotNull(doneLink);
            Assert.AreEqual("Done", doneLink.Target.State);
            var inProgLink = childLinks.FirstOrDefault(cl => cl.Target.Id == inProgTask.Id);
            Assert.IsNotNull(inProgLink);
            Assert.AreEqual("Done", inProgLink.Target.State);
            Assert.AreEqual(0.5, inProgLink.Target.Effort);

            // Assertions on PBI in next sprint

            sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 64.X");
            Item tstNewItem = sm.PBIs.FirstOrDefault
                (pbi => pbi.Title == "[Rolled over] Test of inserted PBI");
            Assert.IsNotNull(tstNewItem);
            Assert.AreEqual(7.0, tstNewItem.Effort);
            Assert.IsTrue(tstNewItem.State != "Done");

            // Assertions on child tasks of PBI in next sprint

            var newChildLinks = sm.ItemAdapter.ItemLinksFromSourceId(tstNewItem.Id);
            Assert.IsNotNull(newChildLinks);
            Assert.AreEqual(2, newChildLinks.Count);
            var toDoLink = newChildLinks.FirstOrDefault
                (cl => cl.Target.Description == "This is the unstarted task");
            Assert.IsNotNull(toDoLink);
            Assert.AreEqual(2.0, toDoLink.Target.RemainingWork);
            Assert.AreEqual(toDoTask.Id, toDoLink.Target.Id);
            var newInProgLink = newChildLinks.FirstOrDefault
                (cl => cl.Target.Description == "[Rolled over] This is the in progress task");
            Assert.IsNotNull(newInProgLink);
            Assert.AreEqual("To Do", newInProgLink.Target.State);
            Assert.AreEqual(1.5, newInProgLink.Target.Effort);
            Assert.AreEqual(1.5, newInProgLink.Target.RemainingWork);
            CleanRollOverItems();
        }

        [TestMethod]
        public void RollPartCompletePBIToBacklog()
        {
            CleanRollOverItems();

            // Create the parent PBI in the previous print

            Item oldItem = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is a demo of a PBI insertion",
                Effort = 12.0,
                IterationPath = "DataWarehouse\\Sprint 64",
                State = "New",
                Title = "Test of inserted PBI",
                WorkItemType = "Product Backlog Item"
            };
            string err = WIA.InsertItem(oldItem);
            Assert.AreEqual(string.Empty, err);
            oldItem.State = "Approved";
            err = WIA.UpdateItem(oldItem);
            Assert.AreEqual(string.Empty, err);
            oldItem.State = "Committed";
            err = WIA.UpdateItem(oldItem);
            Assert.AreEqual(string.Empty, err);

            // Create a task that will be in the 'Done' state
            // on sprint roll over

            Item doneTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is the done task",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 64",
                RemainingWork = 2.0,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(doneTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(oldItem, doneTask, "Child");
            Assert.AreEqual(string.Empty, err);

            // Advance through states

            doneTask.State = "In Progress";
            err = WIA.UpdateItem(doneTask);
            Assert.AreEqual(string.Empty, err);
            doneTask.RemainingWork = 0;
            doneTask.State = "Done";
            err = WIA.UpdateItem(doneTask);
            Assert.AreEqual(string.Empty, err);

            // Create a task that will be in the
            // 'In Progress' state on sprint roll over

            Item inProgTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is the in progress task",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 64",
                RemainingWork = 2.0,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(inProgTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(oldItem, inProgTask, "Child");
            Assert.AreEqual(string.Empty, err);

            // Advance through states

            inProgTask.State = "In Progress";
            inProgTask.RemainingWork = 1.5;
            err = WIA.UpdateItem(inProgTask);
            Assert.AreEqual(string.Empty, err);

            // Create a task that will be in the
            // 'To Do' state on sprint roll over

            Item toDoTask = new Item
            {
                Id = 0,
                AcceptanceCriteria = "GIVEN Pre WHEN Event THEN Post",
                AreaPath = "DataWarehouse\\Stream 5",
                AssignedTo = "Peter Gadsby",
                Description = "This is the unstarted task",
                Effort = 2.0,
                IterationPath = "DataWarehouse\\Sprint 64",
                RemainingWork = 2.0,
                State = "To Do",
                Title = "Test of inserted Task",
                WorkItemType = "Task"
            };
            err = WIA.InsertItem(toDoTask);
            Assert.AreEqual(string.Empty, err);
            err = WIA.InsertLink(oldItem, toDoTask, "Child");
            Assert.AreEqual(string.Empty, err);

            // Now roll this story over

            SprintManager sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 64");
            sm.RollOver(oldItem, false, null);

            // Explore sprint 64 stream 5 for output items

            sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 64");

            // Assertions on parent PBI in previous sprint

            Item tstOldItem = sm.PBIs.FirstOrDefault(pbi => pbi.Id == oldItem.Id);
            Assert.IsNotNull(tstOldItem);
            Assert.AreEqual(5.0, tstOldItem.Effort);
            Assert.AreEqual("Done", tstOldItem.State);

            // Assertions on tasks in previous sprint

            var childLinks = sm.ItemAdapter.ItemLinksFromSourceId(tstOldItem.Id);
            Assert.IsNotNull(childLinks);
            Assert.AreEqual(2, childLinks.Count);
            var doneLink = childLinks.FirstOrDefault(cl => cl.Target.Id == doneTask.Id);
            Assert.IsNotNull(doneLink);
            Assert.AreEqual("Done", doneLink.Target.State);
            var inProgLink = childLinks.FirstOrDefault(cl => cl.Target.Id == inProgTask.Id);
            Assert.IsNotNull(inProgLink);
            Assert.AreEqual("Done", inProgLink.Target.State);
            Assert.AreEqual(0.5, inProgLink.Target.Effort);

            // Assertions on PBI in product backlog

            sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse");
            Item tstNewItem = sm.PBIs.FirstOrDefault
                (pbi => pbi.Title == "[Rolled over] Test of inserted PBI");
            Assert.IsNotNull(tstNewItem);
            Assert.AreEqual(7.0, tstNewItem.Effort);
            Assert.IsTrue(tstNewItem.State != "Done");

            // Assertions on child tasks of PBI in next sprint

            var newChildLinks = sm.ItemAdapter.ItemLinksFromSourceId(tstNewItem.Id);
            Assert.IsNotNull(newChildLinks);
            Assert.AreEqual(2, newChildLinks.Count);
            var toDoLink = newChildLinks.FirstOrDefault
                (cl => cl.Target.Description == "This is the unstarted task");
            Assert.IsNotNull(toDoLink);
            Assert.AreEqual(2.0, toDoLink.Target.RemainingWork);
            Assert.AreEqual(toDoTask.Id, toDoLink.Target.Id);
            var newInProgLink = newChildLinks.FirstOrDefault
                (cl => cl.Target.Description == "[Rolled over] This is the in progress task");
            Assert.IsNotNull(newInProgLink);
            Assert.AreEqual("To Do", newInProgLink.Target.State);
            Assert.AreEqual(1.5, newInProgLink.Target.Effort);
            Assert.AreEqual(1.5, newInProgLink.Target.RemainingWork);
            CleanRollOverItems();
        }

        [TestMethod]
        public void SprintAfter()
        {
            SprintManager sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 63");
            string nextIteration = sm.SprintAfter();
            Assert.AreEqual("DataWarehouse\\Sprint 64", nextIteration);
        }

        [TestMethod]
        public void NonNumericSprintAfter()
        {
            SprintManager sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 64");
            string nextIteration = sm.SprintAfter();
            Assert.AreEqual("DataWarehouse\\Sprint 64.X", nextIteration);
        }

        [TestMethod]
        public void NumericSprintAfter()
        {
            SprintManager sm = new SprintManager
                (WIA, "DataWarehouse\\Stream 5", "DataWarehouse\\Sprint 64.X");
            string nextIteration = sm.SprintAfter();
            Assert.AreEqual("DataWarehouse\\Sprint 65", nextIteration);
        }

        [TestMethod]
        public void GetChildItems()
        {
            var childTasks = WIA.FindLinkedItems(66249, lt => lt == "Child", it => it == "Task");
            Assert.AreEqual(2, childTasks.Count());
            Assert.AreEqual(49233, childTasks.First().Id);
            Assert.AreEqual(49234, childTasks.Skip(1).First().Id);
        }

        [TestMethod]
        public void AllCurrentWorkItems()
        {
            Assert.IsTrue(WIA.AllCurrentWorkItems.Count() > 0);
        }

        [TestMethod]
        public void AllCurrentLinkInfos()
        {
            Assert.IsTrue(WIA.AllCurrentWorkItemLinks.Count() > 0);
        }

        [TestMethod]
        public void AllDescendants()
        {
            Item parentEpic = WIA.AllCurrentWorkItems.FirstOrDefault(wi => wi.Id == 44959);
            Assert.IsNotNull(parentEpic);
            var childItems = WIA.FindDescendants(parentEpic, null);
            Assert.AreEqual(31, childItems.Count());
        }

        [TestMethod]
        public void FilteredDescendants()
        {
            Item parentEpic = WIA.AllCurrentWorkItems.FirstOrDefault(wi => wi.Id == 44959);
            Assert.IsNotNull(parentEpic);
            var childItems = WIA.FindDescendants(parentEpic, it => it == "Product Backlog Item");
            Assert.AreEqual(9, childItems.Count());
        }

        [TestMethod]
        public void GetAssemblies()
        {
            var assemblies = RulesEngine.GetAssemblies();
            Assert.IsTrue(assemblies.Count() == 1);
            Assert.IsTrue(assemblies.Any(a => a.FullName.Contains("TFSLib")));
        }

        [TestMethod]
        public void GetRuleClasses()
        {
            var ruleClasses = RulesEngine.GetRuleClasses();
            Assert.AreEqual(1, ruleClasses.Count());
            Assert.IsTrue(ruleClasses.Any(rc => rc.Name.Contains("RuleValidator")));
        }

        [TestMethod]
        public void GetRuleMethods()
        {
            var ruleClasses = RulesEngine.GetRuleClasses();
            Assert.AreEqual(1, ruleClasses.Count());
            var ruleDelegates = RulesEngine.GetWorkItemRules
                (Activator.CreateInstance(ruleClasses.First(), new object[] { WIA }));
            Assert.IsTrue(ruleDelegates.Count() > 0);
        }

        [TestMethod]
        public void RunFirstRule()
        {
            var ruleClasses = RulesEngine.GetRuleClasses();
            Assert.AreEqual(1, ruleClasses.Count());
            var ruleDelegates = RulesEngine.GetWorkItemRules
                (Activator.CreateInstance(ruleClasses.First(), new object[] { WIA }));
            var violation = ruleDelegates.First()
                (WIA.AllCurrentWorkItems.FirstOrDefault(wi => wi.Id == 44959));
            Assert.IsNull(violation);
        }

        [TestMethod]
        public void EpicCannotBeInAnIteration()
        {
            RuleValidator rv = new RuleValidator(WIA);
            var violation = rv.EpicCannotBeInAnIteration
                (WIA.AllCurrentWorkItems.FirstOrDefault(wi => wi.Id == 47181));
            Assert.IsNotNull(violation);
        }

        [TestMethod]
        public void DescriptionCannotBeEmpty()
        {
            RuleValidator rv = new RuleValidator(WIA);
            var violation = rv.DescriptionCannotBeEmpty
                (WIA.AllCurrentWorkItems.FirstOrDefault(wi => wi.Id == 43691));
            Assert.IsNotNull(violation);
        }

        [TestMethod]
        public void TitleCannotBeEmpty()
        {
            RuleValidator rv = new RuleValidator(WIA);
            var violation = rv.TitleCannotBeEmpty
                (WIA.AllCurrentWorkItems.FirstOrDefault(wi => wi.Id == 18188));
            Assert.IsNull(violation);
        }

        [TestMethod]
        public void TestLinks()
        {
            var parentLinks = WIA.AllCurrentWorkItemLinks
                .Where(il => il.Source.Id == 69007)
                .ToList();
            Assert.IsNotNull(parentLinks);
            Assert.AreEqual(1, parentLinks.Count);
            Assert.AreEqual("Parent", parentLinks[0].LinkType);
            var childLinks = WIA.AllCurrentWorkItemLinks
                .Where(il => il.Source.Id == parentLinks[0].Target.Id)
                .ToList();
            Assert.IsNotNull(childLinks);
            Assert.IsTrue(childLinks.Count > 1);
            var childLink = childLinks.FirstOrDefault
                (il => il.LinkType == "Child" && il.Target.Id == 69007);
            Assert.IsNotNull(childLink);
        }

        [TestMethod]
        public void MustHaveParent()
        {
            //44520
            RuleValidator rv = new RuleValidator(WIA);
            var violation = rv.OnlyEpicsCanHaveNoParent
                (WIA.AllCurrentWorkItems.FirstOrDefault(wi => wi.Id == 69007));
            Assert.IsNull(violation);
        }

        [TestMethod]
        public void ChildrenOfNonDefectEpicMustBeFeaturesOrPBIs()
        {
            RuleValidator rv = new RuleValidator(WIA);
            var violation = rv.ChildrenOfNonDefectEpicMustBeFeaturesOrPBIs
            (
                WIA.AllCurrentWorkItems.FirstOrDefault(wi => wi.Id == 44252),
                "Child",
                WIA.AllCurrentWorkItems.FirstOrDefault(wi => wi.Id == 44898)
            );
            Assert.IsNotNull(violation);
        }

        [TestMethod]
        public void RunRulesEngine()
        {
            var violations = RulesEngine.Run(WIA);
            Assert.IsTrue(violations.Count() > 0);
        }
    }
}
