﻿/*
 * Author: 
 *  S D Smith
 *  
 * Date:
 *  Oct 2015
 *  
 * Purpose: 
 *  Data access layer for loading work items and links from TFS. Used as
 *  part of the TFS sanity checks to make sure TFS users in Analytics
 *  conform to recommended best practices in using the work items to
 *  plan and track backlogs and sprints.
 */

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.TeamFoundation.Client;
using Microsoft.TeamFoundation.Framework.Common;
using Microsoft.TeamFoundation.Framework.Client;
using Microsoft.TeamFoundation.WorkItemTracking.Client;
using System.Text.RegularExpressions;
using System.Collections.ObjectModel;
using System.Xml.Linq;
using System.Xml;
using Microsoft.TeamFoundation.Server;

namespace TFSLib
{
    public class WorkItemAdapter
    {
        private static Dictionary<string, WorkItemAdapter> sharedAdapters = null;

        /// <summary>
        /// Factory method for accessing the work item dapter for a project
        /// </summary>
        /// <param name="project">The name of the TFS team project</param>
        /// <returns>A work item adapter for the selected project</returns>
        
        public static WorkItemAdapter ItemAdapter(string project)
        {
            WorkItemAdapter wia = null;
            if (sharedAdapters == null)
                sharedAdapters = new Dictionary<string, WorkItemAdapter>();
            sharedAdapters.TryGetValue(project, out wia);
            if (wia == null)
            {
                wia = new WorkItemAdapter("http://tfs:8080/tfs/FTPDev/" + project);
                sharedAdapters[project] = wia;
            }
            return wia;
        }

        /// <summary>
        /// Delete the work item adapter for the named project, if it
        /// exists. This ensures that next time it is accessed, the
        /// cached details for the named project will be reloaded.
        /// </summary>
        /// <param name="project">The name of the TFS team project</param>
        
        public static void RefreshAdapter(string project)
        {
            WorkItemAdapter wia = null;
            if (sharedAdapters == null)
                sharedAdapters = new Dictionary<string, WorkItemAdapter>();
            sharedAdapters.TryGetValue(project, out wia);
            if (wia != null)
                sharedAdapters[project] = null;
        }

        /// <summary>
        /// The name of the TFS project. Used in queries
        /// as the root of an iteration or area path
        /// </summary>
        
        public string TfsProjectName
        {
            get;
            private set;
        }

        private string tfsServerName;
        private string tfsCollectionName;
        private TfsTeamProjectCollection tfsProjColl;
        private WorkItemStore tfsWorkItemStore;
        private Project tfsProject;
        private Dictionary<int, WorkItemLinkType> tfsLinkTypes;
        private ICommonStructureService4 commonStructureService;
        private TfsTeamService teamService;

        /// <summary>
        /// All unremoved work items
        /// </summary>
        
        public List<Item> AllCurrentWorkItems
        {
            get;
            private set;
        }

        /// <summary>
        /// Quick lookup for a work item from its ID
        /// </summary>
        
        private Dictionary<int, Item> workItemFromId = null;

        /// <summary>
        /// Obtain the work item that has the specified ID,
        /// or null if the ID is invalid
        /// </summary>
        /// <param name="id">ID of item to look up</param>
        /// <returns>The found item with the ID, or null
        /// if not found</returns>
        
        public Item ItemFromId(int id)
        {
            Item wi;
            workItemFromId.TryGetValue(id, out wi);
            return wi;
        }

        private void initAllCurrentWorkItems()
        {
            AllCurrentWorkItems = loadWorkItems(null);
            workItemFromId = new Dictionary<int, Item>();
            foreach (Item wi in AllCurrentWorkItems)
                workItemFromId.Add(wi.Id, wi);
        }

        private void cacheNewWorkItem(Item item)
        {
            if (!AllCurrentWorkItems.Any(i => i.Id == item.Id))
            {
                AllCurrentWorkItems.Add(item);
                workItemFromId.Add(item.Id, item);
            }
        }

        /// <summary>
        /// All links between non-removed objects
        /// </summary>
        
        public List<ItemLink> AllCurrentWorkItemLinks
        {
            get;
            private set;
        }

        private Dictionary<int, List<ItemLink>> currentItemLinks = null;

        private void initCurrentWorkItemLinks()
        {
            AllCurrentWorkItemLinks = SelectLinks(null);
            currentItemLinks = new Dictionary<int, List<ItemLink>>();
            foreach (var gq in AllCurrentWorkItemLinks.GroupBy(il => il.Source.Id))
                currentItemLinks.Add(gq.Key, gq.ToList());
        }

        /// <summary>
        /// Look up links that have a particular source work item
        /// </summary>
        
        public List<ItemLink> ItemLinksFromSourceId(int id)
        {
            List<ItemLink> links = null;
            if(currentItemLinks.TryGetValue(id, out links))
                return links;
            else return new List<ItemLink>();
        }

        private Item sourceFor(WorkItemLinkInfo wili)
        {
            return ItemFromId(wili.SourceId);
        }

        private Item targetFor(WorkItemLinkInfo wili)
        {
            return ItemFromId(wili.TargetId);
        }

        private string linkTypeFor(WorkItemLinkInfo wili)
        {
            if (wili.LinkTypeId > 0)
                return tfsLinkTypes[wili.LinkTypeId].ForwardEnd.Name;
            else
                return tfsLinkTypes[-wili.LinkTypeId].ReverseEnd.Name;
        }

        // Given the name of a link type, look up the actual TFS
        // work item link type associated with the name

        private WorkItemLinkType LinkTypeFromName(string name)
        {
            foreach (WorkItemLinkType wilt in tfsLinkTypes.Values)
                if (string.Compare(wilt.ForwardEnd.Name, name, true) == 0
                    || string.Compare(wilt.ReverseEnd.Name, name, true) == 0)
                    return wilt;
            return null;
        }

        // Find out if the specified name is the forward
        // direction link name for a work item link
        
        private bool IsForwardLinkName(WorkItemLinkType wilt, string name)
        {
            return string.Compare(wilt.ForwardEnd.Name, name, true) == 0;
        }

        /// <summary>
        /// Constructor. This is private in order to force users to
        /// generate work item adapters using the factory method:
        /// WorkItemAdapter ItemAdapter(string teamProjectName)
        /// </summary>
        /// <param name="tfsProjectUrl">The URL down to the project name,
        /// of the form: http://tfsMachine:8080/tfs/CollectionName/ProjectName 
        /// </param>
        
        private WorkItemAdapter(string tfsProjectUrl)
        {
            TfsProject = tfsProjectUrl;
            tfsLinkTypes = loadLinkTypes();
            initAllCurrentWorkItems();
            initCurrentWorkItemLinks();
        }

        /// <summary>
        /// Get/set project URI of the form: http://tfsMachine:8080/tfs/CollectionName/ProjectName
        /// </summary>
        
        public string TfsProject
        {
            get
            {
                return 
                    tfsProject == null 
                    ? string.Empty 
                    : (tfsServerName + "/" + tfsCollectionName + "/" + TfsProjectName);
            }

            private set
            {
                if(!string.IsNullOrEmpty(value))
                {
                    parseUri(value);

                    // Load the project collection

                    tfsProjColl = 
                        TfsTeamProjectCollectionFactory.GetTeamProjectCollection
                            (new Uri(tfsServerName + "/" + tfsCollectionName));

                    tfsWorkItemStore = tfsProjColl.GetService<WorkItemStore>();
                    tfsProject = tfsWorkItemStore.Projects[TfsProjectName];
                    commonStructureService = tfsProjColl.GetService<ICommonStructureService4>();
                    teamService = tfsProjColl.GetService<TfsTeamService>();
                }

                if (tfsProject == null || tfsWorkItemStore == null)
                    throw new ArgumentException("Invalid TFS project URL constructing WorkItemAdapter");
            }
        }

        private static Regex tfsUriRe = new Regex("^(http://[\\w\\.]+:8080/\\w+)/(\\w+)/(\\w+)$");
        private void parseUri(string uri)
        {
            Match m = tfsUriRe.Match(uri);
            if(!m.Success)
                throw new ArgumentException("Badly formed TFS URI. Good example: http://TfsServerName:8080/TfsVirtDir/CollectionName/ProjectName");

            tfsServerName = m.Groups[1].Value;
            tfsCollectionName = m.Groups[2].Value;
            TfsProjectName = m.Groups[3].Value;
        }

        /// <summary>
        /// Given an iteration path, find the start and end dates for that print
        /// </summary>
        /// <param name="iterationPath">The iteration path truncated to remove the
        /// collection and project, for example, "Sprint 63"</param>
        /// <returns>The start and end dates</returns>
        
        public ScheduleInfo GetSprintSchedule(string iterationPath)
        {
            return SprintSchedules
                .FirstOrDefault(si => si.IterationPath == iterationPath);
        }

        private List<ScheduleInfo> sprintSchedules = null;

        /// <summary>
        /// Get the complete set of sprint schedules, for sprints
        /// that have at least either a start date or an end date
        /// </summary>
        /// <returns>The list of sprints</returns>
        
        public List<ScheduleInfo> SprintSchedules
        {
            get
            {
                if (sprintSchedules == null)
                {
                    var projectInfo = commonStructureService.GetProjectFromName(TfsProjectName);
                    NodeInfo[] structures = commonStructureService.ListStructures(projectInfo.Uri);
                    var iterations = structures.FirstOrDefault(s => s.StructureType.Equals("ProjectLifecycle"));

                    if (iterations != null)
                    {
                        XmlElement iterationsTree = commonStructureService.GetNodesXml(new[] { iterations.Uri }, true);
                        sprintSchedules = WalkTreeForIterations(iterationsTree);
                    }
                }
                sprintSchedules.Sort
                (
                    (s, t) => 
                        Math.Sign
                        (
                            s.StartDate.GetValueOrDefault(DateTime.MinValue).Ticks 
                                - t.StartDate.GetValueOrDefault(DateTime.MinValue).Ticks
                        )
                );
                return sprintSchedules;
            }
        }

        private Dictionary<string, List<string>> users = null;

        /// <summary>
        /// Obtain the list of TFS teams and user names in the selected project
        /// </summary>
        
        public Dictionary<string, List<string>> Teams
        {
            get
            {
                if(users == null)
                {
                    users = new Dictionary<string, List<string>>();
                    var projectInfo = commonStructureService.GetProjectFromName(TfsProjectName);
                    var allTeams = teamService.QueryTeams(projectInfo.Uri);
                    foreach(var team in allTeams)
                    {
                        users[team.Name] = new List<string>();
                        IEnumerable<TeamFoundationIdentity> identities =
                            team.GetMembers(tfsProjColl, MembershipQuery.Expanded).Where(m=>!m.IsContainer);
                        foreach (string userName in identities.Select(i => i.DisplayName))
                            if (!users[team.Name].Contains(userName))
                                users[team.Name].Add(userName);
                    }
                }
                return users;
            }
        }

        /// <summary>
        /// Populate the history of an Item from its revisions
        /// </summary>
        /// <param name="i">The item to populate</param>
        
        public void PopulateHistory(Item i)
        {
            if (i.History == null)
                i.History = ItemHistory(i);
        }

        private List<ScheduleInfo> WalkTreeForIterations(XmlElement iterNode)
        {
            List<ScheduleInfo> iterationSchedule = new List<ScheduleInfo>();
            string startDate = iterNode.GetAttribute("StartDate");
            string endDate = iterNode.GetAttribute("FinishDate");
            string iterPath = iterNode.GetAttribute("Path");

            string iterPreamble = string.Format("\\{0}\\Iteration\\", TfsProjectName);
            if (iterPath != null && iterPath.StartsWith
                (iterPreamble, StringComparison.CurrentCultureIgnoreCase))
                iterPath = iterPath.Substring(iterPreamble.Length);
            else
                iterPath = null;

            if (!string.IsNullOrEmpty(iterPath) &&
                (!string.IsNullOrEmpty(startDate)
                || !string.IsNullOrEmpty(endDate)))
            {
                ScheduleInfo si = new ScheduleInfo();
                si.IterationPath = TfsProjectName + "\\" + iterPath;
                DateTime t;
                if (DateTime.TryParse(startDate, out t))
                    si.StartDate = t;
                else
                    si.StartDate = null;
                if (DateTime.TryParse(endDate, out t))
                    si.EndDate = t;
                else
                    si.EndDate = null;
                iterationSchedule.Add(si);
            }
            if (iterNode.HasChildNodes)
                foreach (XmlElement e in iterNode.ChildNodes)
                    iterationSchedule.AddRange(WalkTreeForIterations(e));
            return iterationSchedule;
        }

        /// <summary>
        /// Search for work items in the TFS database. The
        /// three filters that can be applied are for the item type,
        /// the iteration name and the team name
        /// </summary>
        /// <param name="itemType">The type of work item in TFS. Can
        /// be one of "Epic", "Feature", "Product Backlog Item",
        /// "Bug", or "Task"</param>
        /// <param name="sprint">The sprint path. In the format
        /// "ProjectName\\Sprint 32" for example./param>
        /// <param name="team">The scrum team assigned to
        /// this work item, e.g. "ProjectName\\Stream 1"</param>
        /// <returns>The list of work items that match the
        /// specified search criteria</returns>
        
        public IEnumerable<Item> FindWorkItemsInTeamSprint
            ( string itemType, string sprint, string team )
        {
            var result = AllCurrentWorkItems.AsEnumerable();
            if (!string.IsNullOrEmpty(itemType))
                result = result.Where(wi => wi.WorkItemType == itemType);
            if (!string.IsNullOrEmpty(sprint))
                result = result.Where(wi => wi.IterationPath.StartsWith
                    (sprint, StringComparison.CurrentCultureIgnoreCase));
            if (!string.IsNullOrEmpty(team))
                result = result.Where(wi => wi.AreaPath.StartsWith
                    (team, StringComparison.CurrentCultureIgnoreCase));
            return result;
        }

        /// <summary>
        /// Look up an item direct from the TFS store by ID
        /// </summary>
        /// <param name="id">ID of item to be looked up</param>
        /// <returns>The item with fields of interets to us</returns>
        
        public Item FindItem(int id)
        {
            return AllCurrentWorkItems.FirstOrDefault(i => i.Id == id);
        }

        /// <summary>
        /// Look up the set of items that match the corresponding
        /// TFS query language where clause. WARNING: Bypasses item cache.
        /// </summary>
        /// <param name="whereClause">The TQL where clause</param>
        /// <returns>The set of items matching the query</returns>
        
        public IEnumerable<Item> SelectItemsDirectFromTFS(string whereClause)
        {
            return loadWorkItems(whereClause);
        }

        /// <summary>
        /// Look up the set of items that match the corresponding
        /// predicate. Items are fetched from the in memory cache.
        /// </summary>
        /// <param name="whereClause">The filtering predicate</param>
        /// <returns>The set of items matching the query</returns>

        public IEnumerable<Item> Select(Func<Item, bool> predicate)
        {
            return AllCurrentWorkItems.Where(predicate);
        }

        /// <summary>
        /// Insert a link between two work items
        /// </summary>
        /// <param name="src">The item from which a link is required</param>
        /// <param name="tgt">The item at the other end of the link</param>
        /// <param name="linkName">The role of the item at the target end of
        /// the link, with respect to the source item. E.g. if the source
        /// is to be the parent of the target, the link name will be "Child"</param>
        /// <returns>An error message if not successful, otherwise empty string</returns>
        
        public string InsertLink(Item src, Item tgt, string linkName)
        {
            // Make sure the link type is a valid link type

            WorkItemLinkType wilt = LinkTypeFromName(linkName);
            if (wilt == null)
                return "Unrecognised link type '" 
                    + linkName 
                    + "' inserting link between " 
                    + src.Id 
                    + " and " 
                    + tgt.Id;

            // Make sure this link type does not already
            // exist between these two work items

            var linkedItems = FindLinkedItems(src, ln => ln == linkName, null);
            if(linkedItems.Any(li => li.Id == tgt.Id))
                return "Link of type '" 
                    +linkName 
                    + "' already exists between " 
                    + src.Id 
                    + " and " 
                    + tgt.Id;

            // If the link name is "Parent" then we need to ensure we
            // don't already have another parent link, as we can only
            // have a single parent item

            if(linkName == "Parent" && linkedItems.Any())
                return "Link of type '" 
                    +linkName 
                    + "' already exists between " 
                    + src.Id 
                    + " and another work item"; 

            // Swap source and target if
            // the link name is a reverse name

            if(!IsForwardLinkName(wilt, linkName))
            {
                Item tmp = src;
                src = tgt;
                tgt = tmp;
            }

            // Find the TFS work item that corresponds to the source item

            try
            {
                WorkItem wi = tfsWorkItemStore.GetWorkItem(src.Id);
                if (wi == null)
                    return "Item with ID " + src.Id + " not found in TFS";

                WorkItemLink wiLink = new WorkItemLink(wilt.ForwardEnd, src.Id, tgt.Id);
                wi.WorkItemLinks.Add(wiLink);
                wi.Save();
                ItemLink forwardLink = new ItemLink(src, tgt, wilt.ForwardEnd.Name);
                AllCurrentWorkItemLinks.Add(forwardLink);
                if (!currentItemLinks.Keys.Contains(src.Id))
                    currentItemLinks[src.Id] = new List<ItemLink>();
                currentItemLinks[src.Id].Add(forwardLink);
                ItemLink reverseLink = new ItemLink(tgt, src, wilt.ReverseEnd.Name);
                AllCurrentWorkItemLinks.Add(reverseLink);
                if (!currentItemLinks.Keys.Contains(tgt.Id))
                    currentItemLinks[tgt.Id] = new List<ItemLink>();
                currentItemLinks[tgt.Id].Add(reverseLink);
                return string.Empty;
            }
            catch(Exception x)
            {
                return x.Message;
            }
        }

        private string OppositeDirectionLinkType(string linkType)
        {
            WorkItemLinkType wilt = LinkTypeFromName(linkType);
            if (IsForwardLinkName(wilt, linkType))
                return wilt.ReverseEnd.Name;
            else
                return wilt.ForwardEnd.Name;
        }

        private ItemLink FindReverseLink(ItemLink link)
        {
            string revLinkType = OppositeDirectionLinkType(link.LinkType);
            return AllCurrentWorkItemLinks.FirstOrDefault
            (
                wil => wil.Source.Id == link.Target.Id &&
                wil.Target.Id == link.Source.Id &&
                wil.LinkType == revLinkType
            );
        }

        /// <summary>
        /// Given the work items at each end of a link and the link type,
        /// delete the link between the two work items
        /// </summary>
        /// <param name="src">Work item at the source end of the link</param>
        /// <param name="tgt">Work item at the target end of the link</param>
        /// <param name="linkType">The name of the type of link</param>
        /// <returns>Error message if unable to delete, or empty string if
        /// deletion successful</returns>
        
        public string DeleteLink(Item src, Item tgt, string linkType)
        {
            ItemLink forwardLink = AllCurrentWorkItemLinks.FirstOrDefault
            (
                wil => wil.Source.Id == src.Id &&
                wil.Target.Id == tgt.Id &&
                wil.LinkType == linkType
            );

            if (forwardLink != null)
            {
                ItemLink reverseLink = FindReverseLink(forwardLink);
                if (reverseLink == null)
                    return "Link table damaged - link only found in one direction";
                WorkItem sourceWorkItem = tfsWorkItemStore.GetWorkItem(forwardLink.Source.Id);
                if (sourceWorkItem == null)
                    return "Item with ID " + forwardLink.Source.Id + " not found in TFS";
                WorkItem targetWorkItem = tfsWorkItemStore.GetWorkItem(forwardLink.Target.Id);
                if (targetWorkItem == null)
                    return "Item with ID " + forwardLink.Target.Id + " not found in TFS";
                foreach (WorkItemLink l in sourceWorkItem.WorkItemLinks)
                {
                    if (l.LinkTypeEnd.Name == forwardLink.LinkType && l.TargetId == targetWorkItem.Id)
                    {
                        sourceWorkItem.WorkItemLinks.Remove(l);
                        sourceWorkItem.Save();
                        DeleteLinkCacheEntry(forwardLink);
                        DeleteLinkCacheEntry(reverseLink);
                        return string.Empty;
                    }
                }
            }

            return "Link of type '"
                + forwardLink.LinkType
                + "' between "
                + forwardLink.Source.Id
                + " and "
                + forwardLink.Target.Id
                + " not found";
        }

        private void DeleteLinkCacheEntry(ItemLink link)
        {
            List<ItemLink> linksList = null;
            if(currentItemLinks.TryGetValue(link.Source.Id, out linksList))
            {
                ItemLink deletee = linksList.FirstOrDefault
                    (l => l.Target.Id == link.Target.Id && l.LinkType == link.LinkType);
                if (deletee != null)
                    linksList.Remove(deletee);
            }
            AllCurrentWorkItemLinks.Remove(link);
        }

        /// <summary>
        /// Delete a link between two work items from TFS. Note that any reverse
        /// link of te opposite type is also deleted as they constitute a pair.
        /// </summary>
        /// <param name="forwardLink">The link between two work items. Note that
        /// this doesn't need to be the instance held in the work item adapter.
        /// It's source and target work items and the link type are used to find
        /// the item in the cache.</param>
        /// <returns>An error message if unsuccessful, or string.Empty if OK</returns>
        
        public string DeleteLink(ItemLink forwardLink)
        {
            return DeleteLink(forwardLink.Source, forwardLink.Target, forwardLink.LinkType);
        }

        /// <summary>
        /// Insert a new item into the TFS work item store
        /// </summary>
        /// <param name="item">The item values for each of the work item fields</param>
        /// <returns>An error message or the empty string if successful</returns>
        
        public string InsertItem(Item item)
        {
            try
            {
            string errResult = string.Empty;
            if (item.Id != 0)
                return "New items should not have an ID value set";
            WorkItemType wit = tfsProject.WorkItemTypes[item.WorkItemType];
            if (wit == null)
                return "Unrecognised work item type";
            WorkItem wi = new WorkItem(wit);
            wi.State = item.State;
            if (!wi.Fields["State"].IsValid)
                return "Cannot set state to " + item.State;

            if (string.IsNullOrEmpty(item.Title))
                return "Cannot put an empty title on a workitem";
            else
                wi.Title = item.Title;

            if (string.IsNullOrEmpty(item.Description))
                return "Cannot put an empty description on a workitem";

            if (item.WorkItemType == "Bug")
            {
                errResult = SetField<string>(wi, "Microsoft.VSTS.TCM.ReproSteps", item.Description);
                if (!string.IsNullOrEmpty(errResult))
                    return errResult;
            }
            else
                wi.Description = item.Description;

            if (!item.IterationPath.StartsWith(TfsProjectName, StringComparison.CurrentCultureIgnoreCase))
                return "Sprint iteration must begin with TFS project name";
            else
                wi.IterationPath = item.IterationPath;

            if (!item.AreaPath.StartsWith(TfsProjectName, StringComparison.CurrentCultureIgnoreCase))
                return "Sprint team area must begin with TFS project name";
            else
                wi.AreaPath = item.AreaPath;

            if(item.Effort < 0)
                return "Work items cannot have negative effort";

            string effortFieldName = "Effort";
            if(item.WorkItemType == "Task")
                effortFieldName = "Estimated Effort (Scrum v3)";
            if (item.WorkItemType != "Feature")
            {
                errResult = SetField<double>(wi, effortFieldName, item.Effort);
                if (!string.IsNullOrEmpty(errResult))
                    return errResult;
            }

            if (item.WorkItemType == "Task")
            {
                errResult = SetField<double>(wi, "Remaining Work", item.RemainingWork);
                if (!string.IsNullOrEmpty(errResult))
                    return errResult;
            }

            errResult = SetField<string>(wi, "Assigned To", item.AssignedTo);
            if(!string.IsNullOrEmpty(errResult))
                return errResult;

            if (item.WorkItemType != "Task")
            {
                errResult = SetField<string>(wi, "Acceptance Criteria", item.AcceptanceCriteria);
                if (!string.IsNullOrEmpty(errResult))
                    return errResult;
            }

            ArrayList invalidFields = wi.Validate();
            if(invalidFields != null && invalidFields.Count > 0)
            {
                string validationErr = string.Empty;
                foreach (Field f in invalidFields)
                    validationErr += string.Format
                        ("Invalid field: {0}[{1}]\r\n", f.Name, f.Value);
            }
            wi.Save();
            item.Id = wi.Id;
            item.Dirty = false;
            cacheNewWorkItem(item);
            return string.Empty;
            }
            catch(Exception x)
            {
                return "Unhandled exception: "+ x.Message;
            }
        }

        /// <summary>
        /// Flush a work item to TFS if it already existed and has been changed
        /// </summary>
        /// <param name="item">The item as represented in this application</param>
        /// <returns>Empty string if successful, error message if not</returns>
        
        public string UpdateItem(Item item)
        {
            // If the item is not changed, just return without saving

            if (!item.Dirty)
                return string.Empty;
            if (!AllCurrentWorkItems.Contains(item))
                return "Attempting to update an item that was not fetched from TFS";

            // Find the TFS work item that corresponds to this item
            
            WorkItem wi = tfsWorkItemStore.GetWorkItem(item.Id);
            if (wi == null)
                return "Item with ID " + item.Id + " not found in TFS";

            // Apply each member to the work item including performing validation

            if (item.WorkItemType != wi.Type.Name)
                return "Cannot change a work item type from " + wi.Type.Name + " to " + item.WorkItemType;

            if(item.State != wi.State)
            {
                wi.State = item.State;
                if (!wi.Fields["State"].IsValid)
                    return "Cannot change state to " + item.State;
            }

            if(string.IsNullOrEmpty(item.Title))
                return "Cannot put an empty title on a workitem";

            if (item.Title != wi.Title)
                wi.Title = item.Title;

            if (string.IsNullOrEmpty(item.Description))
                return "Cannot put an empty description on a workitem";

            if(item.WorkItemType == "Bug" && item.Description 
                != GetField<string>(wi, "Microsoft.VSTS.TCM.ReproSteps"))
            {
                string result = SetField<string>(wi, "Microsoft.VSTS.TCM.ReproSteps", item.Description);
                if(!string.IsNullOrEmpty(result))
                    return result;
            }

            if (item.WorkItemType != "Bug" && item.Description != wi.Description)
                wi.Description = item.Description;

            if (!item.IterationPath.StartsWith(TfsProjectName, StringComparison.CurrentCultureIgnoreCase))
                return "Sprint iteration must begin with TFS project name";

            if (item.IterationPath != wi.IterationPath)
                wi.IterationPath = item.IterationPath;

            if (!item.AreaPath.StartsWith(TfsProjectName, StringComparison.CurrentCultureIgnoreCase))
                return "Sprint team area must begin with TFS project name";

            if (item.AreaPath != wi.AreaPath)
                wi.AreaPath = item.AreaPath;

            if(item.Effort < 0)
                return "Work items cannot have negative effort";

            string effortFieldName = "Effort";
            if(item.WorkItemType == "Task")
                effortFieldName = "Estimated Effort (Scrum v3)";
            if(item.Effort != GetField<double>(wi, effortFieldName))
            {
                string result = SetField<double>(wi, effortFieldName, item.Effort);
                if(!string.IsNullOrEmpty(result))
                    return result;
            }

            if (item.WorkItemType == "Task")
            {
                if (item.RemainingWork != GetField<double>(wi, "Remaining Work"))
                {
                    string result = SetField<double>(wi, "Remaining Work", item.RemainingWork);
                    if (!string.IsNullOrEmpty(result))
                        return result;
                }
            }

            if(item.AssignedTo != GetField<string>(wi, "Assigned To"))
            {
                string result = SetField<string>(wi, "Assigned To", item.AssignedTo);
                if(!string.IsNullOrEmpty(result))
                    return result;
            }

            if(item.WorkItemType != "Task" 
                && item.AcceptanceCriteria != GetField<string>(wi, "Acceptance Criteria"))
            {
                string result = SetField<string>(wi, "Acceptance Criteria", item.AcceptanceCriteria);
                if(!string.IsNullOrEmpty(result))
                    return result;
            }

            // Do some last minute validation

            ArrayList invalidFields = wi.Validate();
            if(invalidFields != null && invalidFields.Count > 0)
            {
                string result = wi.Type.Name + " " + wi.Id + " validation errors:\r\n";
                foreach(Field f in invalidFields)
                {
                    result += f.Name + " " + f.Status.ToString("G") + "\r\n";
                }
                return result;
            }

            // Now save the changed item to the TFS database

            wi.Save();
            item.Dirty = false;
            return string.Empty;
        }

        /// <summary>
        /// Delete an item from TFS
        /// </summary>
        /// <param name="item">The in memory copy of an item fetched from TFS</param>
        /// <returns>An error message, or string.Empty if successful</returns>

        public string DeleteItem(Item item, bool trueDeletion)
        {
            if (!AllCurrentWorkItems.Contains(item))
                return "Attempting to delete an item that was not fetched from TFS";

            if (!trueDeletion)
                return RemoveItem(item);

            // Go ahead and destroy the work item

            try
            {
                IEnumerable<WorkItemOperationError> errors =
                    tfsWorkItemStore.DestroyWorkItems(new int[] { item.Id });
                string errResult = string.Empty;
                foreach (WorkItemOperationError error in errors)
                    errResult += string.Format("{0}: {1}\r\n", error.Id, error.Exception.Message);
                if (string.IsNullOrEmpty(errResult))
                {
                    workItemFromId.Remove(item.Id);
                    AllCurrentWorkItems.Remove(item);
                }
                return errResult;
            }
            catch (Exception x)
            {
                return x.Message;
            }
        }

        private string SetState(Item i, string state)
        {
            i.State = state;
            string err = UpdateItem(i);
            if (!string.IsNullOrEmpty(err))
                return string.Format("SetState({0}, {1}): {2}\r\n", i.Id, state, err);
            else return string.Empty;
        }

        public string RemoveItem(Item item)
        {
            string err = string.Empty;
            switch(item.WorkItemType)
            {
                case "Feature":
                    if (item.State == "Done")
                        err += SetState(item, "In Progress");
                    if (item.State == "In Progress")
                        err += SetState(item, "New");
                    if (item.State == "New")
                        err += SetState(item, "Removed");
                    break;

                case "Bug":
                case"Product Backlog Item":
                    if (item.State == "Done")
                        err += SetState(item, "Committed");
                    if (item.State == "Committed")
                        err += SetState(item, "Approved");
                    if (item.State == "Approved")
                        err += SetState(item, "New");
                    if (item.State == "New")
                        err += SetState(item, "Removed");
                    break;

                case "Task":
                    if (item.State == "Done")
                    {
                        item.RemainingWork = 0.01;
                        err += SetState(item, "In Progress");
                    }
                    if (item.State == "In Progress")
                        err += SetState(item, "To Do");
                    if (item.State == "To Do")
                    {
                        item.RemainingWork = 0.0;
                        err += SetState(item, "Removed");
                    }
                    break;
            }

            if (string.IsNullOrEmpty(err))
            {
                workItemFromId.Remove(item.Id);
                AllCurrentWorkItems.Remove(item);
            }
            return err;
        }

        private List<Item> loadWorkItems(string whereClause)
        {
            List<Item> foundItems = new List<Item>();
            string query =
                "SELECT [System.Id], [System.State], [System.Title], [System.Description], [System.ChangedBy]," +
                "[System.IterationPath], [System.AreaPath], [System.AssignedTo], [System.ChangedDate], " +
                "[Microsoft.VSTS.Common.AcceptanceCriteria], [Microsoft.VSTS.Scheduling.Effort], [Microsoft.VSTS.TCM.ReproSteps], " +
                "[Microsoft.VSTS.Scheduling.RemainingWork], [System.WorkItemType] FROM WorkItems " +
                "WHERE ([System.TeamProject] = '" + TfsProjectName + "') " +
                "AND ([System.State] NOT IN ('Removed', 'Deleted', 'Closed')) ";
            if (!string.IsNullOrEmpty(whereClause))
                query += "AND (" + whereClause + ") ";
            query += "ORDER BY [System.ChangedDate] DESC";
            WorkItemCollection items = tfsWorkItemStore.Query(query);
            items.PageSize = 200;
            foreach (WorkItem i in items)
                foundItems.Add(BuildItemFromWorkItem(i));
            return foundItems;
        }

        /// <summary>
        /// Build a list of items representing the
        /// history of the item whose ID has been passed
        /// </summary>
        /// <param name="item">A current item whose ID will
        /// be used to look up its history</param>
        /// <returns>The set of items with this ID over time</returns>
        
        public List<Item> ItemHistory(Item item)
        {
            if (!AllCurrentWorkItems.Contains(item))
                throw new ArgumentException("Attempting to update an item that was not fetched from TFS");

            return ItemHistory(item.Id);
        }

        /// <summary>
        /// Build a list of items representing the
        /// history of the item whose ID has been passed
        /// </summary>
        /// <param name="id">The item's ID</param>
        /// <returns>The list of all past revisions of the item</returns>
        
        public List<Item> ItemHistory(int id)
        {
            // Find the TFS work item that corresponds to this item

            WorkItem wi = tfsWorkItemStore.GetWorkItem(id);
            if (wi == null)
                throw new ArgumentException("Item with ID " + id + " not found in TFS");

            List<Item> itemHistory = new List<Item>();
            foreach (Revision rev in wi.Revisions)
                itemHistory.Add(BuildItemFromRevision(rev));
            itemHistory.Sort
                ((i, j) => Math.Sign(i.ChangedDate.Ticks - j.ChangedDate.Ticks));
            return itemHistory;
        }

        private Item BuildItemFromRevision(Revision rev)
        {
            Item item = new Item
            {
                Id = GetField<int>(rev, "ID"),
                WorkItemType = GetField<string>(rev, "Work Item Type"),
                AcceptanceCriteria = GetField<string>(rev, "Acceptance Criteria"),
                AreaPath = GetField<string>(rev, "Area Path"),
                AssignedTo = GetField<string>(rev, "Assigned To"),
                ChangedBy = GetField<string>(rev, "Changed By"),
                ChangedDate = GetField<DateTime>(rev, "Changed Date"),
                IterationPath = GetField<string>(rev, "Iteration Path"),
                RemainingWork = GetField<double>(rev, "Remaining Work"),
                State = GetField<string>(rev, "State"),
                Title = GetField<string>(rev, "Title"),
            };
            if(item.WorkItemType == "Task")
                item.Effort = GetField<double>(rev, "Estimated Effort (Scrum v3)");
            else
                item.Effort = GetField<double>(rev, "Effort");

            if(item.WorkItemType == "Bug")
                item.Description = GetField<string>(rev, "Microsoft.VSTS.TCM.ReproSteps");
            else
                item.Description = GetField<string>(rev, "Description");
            item.Dirty = false;
            return item;
        }

        private static Item BuildItemFromWorkItem(WorkItem i)
        {
            string wiType = i.Type.Name;
            double effort;
            if (wiType == "Task")
                effort = GetField<double>(i, "Estimated Effort (Scrum v3)");
            else
                effort = GetField<double>(i, "Effort");

            string description;
            if (wiType == "Bug")
                description = GetField<string>(i, "Microsoft.VSTS.TCM.ReproSteps");
            else
                description = i.Description;

            Item newItem = new Item
            {
                Id = i.Id,
                WorkItemType = wiType,
                AcceptanceCriteria = GetField<string>(i, "Acceptance Criteria"),
                AreaPath = i.AreaPath,
                AssignedTo = GetField<string>(i, "Assigned To"),
                ChangedDate = i.ChangedDate,
                ChangedBy = i.ChangedBy,
                Description = description,
                Effort = effort,
                IterationPath = i.IterationPath,
                RemainingWork = GetField<double>(i, "Remaining Work"),
                State = i.State,
                Title = i.Title
            };
            newItem.Dirty = false;
            return newItem;
        }

        private static T GetField<T>(WorkItem i, string fieldName)
        {
            if (i.Fields.Contains(fieldName))
            {
                object o = i.Fields[fieldName].Value;
                if (o != null)
                    return (T)o;
            }
            return default(T);
        }
        private static T GetField<T>(Revision i, string fieldName)
        {
            if (i.Fields.Contains(fieldName))
            {
                object o = i.Fields[fieldName].Value;
                if (o != null)
                    return (T)o;
            }
            return default(T);
        }

        private static string SetField<T>(WorkItem i, string fieldName, T value)
        {
            if (i.Fields.Contains(fieldName))
                i.Fields[fieldName].Value = value;
            else
                return "Field " + fieldName + 
                    " does not exist in work items of type " + i.Type.Name;

            if (!i.Fields[fieldName].IsValid)
                return "Value " + value + " not valid for field " + fieldName;
            else
                return string.Empty;
        }

        // Unused - used for diagnosing field names in TFS

        private void InsertFieldNames(WorkItem i)
        {
            string wiType = i.Type.Name;

            if (FieldNames == null)
                FieldNames = new Dictionary<string, List<string>>();

            if (!FieldNames.ContainsKey(wiType))
                FieldNames.Add(wiType, new List<string>());

            foreach(Field field in i.Fields)
            {
                if (!FieldNames[wiType].Any(s => s == field.Name))
                    FieldNames[wiType].Add(field.Name);
            }
        }
        
        public Dictionary<string, List<string>> FieldNames
        {
            get;
            private set;
        }

        private Dictionary<int, WorkItemLinkType> loadLinkTypes()
        {
            var linkTypes = new Dictionary<int, WorkItemLinkType>();
	        foreach (var lt in tfsWorkItemStore.WorkItemLinkTypes)
		        linkTypes.Add(lt.ForwardEnd.Id, lt);
            return linkTypes;
        }

        /// <summary>
        /// Obtain the set of links that originate from a specified item.
        /// WARNING: This bypasses the cache and fetches straight from TFS.
        /// Use FindLinkedItems instead for in-cache searches.
        /// </summary>
        /// <param name="src">The item we are looking at links from. If
        /// null, all links in TFS are returned.</param>
        /// <returns>The set of links as determined by the
        /// parameter 'src'</returns>
        
        public List<ItemLink> SelectLinks(Item src)
        {
            var links = new List<ItemLink>();
            string srcCriterion = " <> 0";
            if (src != null)
                srcCriterion = " = " + src.Id;
            string query = "SELECT * FROM WorkItemLinks WHERE " +
                "(Source.[System.TeamProject] = '" + TfsProjectName + "') " +
                "AND (Source.[System.Id] " + srcCriterion + ") AND (Target.[System.Id] <> 0) " +
                "AND (Source.[System.State] NOT IN ('Removed', 'Deleted', 'Closed')) " +
                "AND (Target.[System.State] NOT IN ('Removed', 'Deleted', 'Closed'))";
            Query queryObj = new Query(tfsWorkItemStore, query);
            WorkItemLinkInfo[] linkObjects = queryObj.RunLinkQuery();
            foreach (WorkItemLinkInfo lo in linkObjects)
                if (lo.LinkTypeId != 0)
                {
                    Item linkSrc = sourceFor(lo);
                    Item linkTgt = targetFor(lo);
                    if (linkSrc != null && linkTgt != null)
                        links.Add(new ItemLink(linkSrc, linkTgt, linkTypeFor(lo)));
                }
            return links;
        }

        /// <summary>
        /// Return the set of links between the two specified items.
        /// WARNING: This bypasses the cache and fetches straight from TFS.
        /// Use FindLinkedItems instead for in-cache searches.
        /// </summary>
        /// <param name="src">The source item in the pair between
        /// which we are seeking links</param>
        /// <param name="tgt">The target item in the pair between
        /// which we are seeking links</param>
        /// <returns>All links specifically between these two items</returns>
        
        public List<ItemLink> SelectLinksBetween(Item src, Item tgt)
        {
            return new List<ItemLink>
                (SelectLinks(src).Where(l => l.Target.Id == tgt.Id));
        }

        /// <summary>
        /// Find the set of work items that are linked to another item
        /// by some standard link type.
        /// </summary>
        /// <param name="id">The System.Id value for the item whose
        /// linked work items we want to identify</param>
        /// <param name="ltPredicate">A predicate function that tests the
        /// type name of the link. The type of link can be one of:
        /// "Related", "Parent", "Child", "Predecessor", "Successor",
        /// "Shared Steps", "Test Case", "Tests", "Tested By", "Fails",
        /// "Failed By", "Implements", "Implemented By", "Impedes",
        /// "Impeded By", "Acceptance Tests", "Accepted By", "References",
        /// "Referenced By"</param>
        /// <param name="wiTypePredicate">A predicate function that tests
        /// the type of work item linked to. Linked item types can be:
        /// one of: "Epic", "Feature", "Product Backlog Item",
        /// "Bug", or "Task"</param>
        /// <returns>The set of linked items matching the search criteria</returns>

        public IEnumerable<Item> FindLinkedItems
            (int id, Func<string, bool> ltPredicate, Func<string, bool> wiTypePredicate)
        {
            // Apply the source and link type filters to choose links
            // of the specified type from the source ID work item

            IEnumerable<ItemLink> links = ItemLinksFromSourceId(id);
            if (ltPredicate != null)
                links = links.Where(wil => ltPredicate(wil.LinkType));
            
            // Find the set of target items that have this source item
            // and the specified link type. Filter on the target work
            // item type too, if a type has been specified.
            
            IEnumerable<Item> linkedItems = links
                .Select(wil => wil.Target);
            if (wiTypePredicate != null)
                linkedItems = linkedItems.Where(wi => wiTypePredicate(wi.WorkItemType));
            return linkedItems;
        }

        /// <summary>
        /// Find the set of work items that are linked to another item
        /// by some standard link type.
        /// </summary>
        /// <param name="wi">The item for which we want
        /// to identify linked work items</param>
        /// <param name="ltPredicate">A predicate function that tests the
        /// type name of the link. The type of link can be one of:
        /// "Related", "Parent", "Child", "Predecessor", "Successor",
        /// "Shared Steps", "Test Case", "Tests", "Tested By", "Fails",
        /// "Failed By", "Implements", "Implemented By", "Impedes",
        /// "Impeded By", "Acceptance Tests", "Accepted By", "References",
        /// "Referenced By"</param>
        /// <param name="wiTypePredicate">A predicate function that tests
        /// the type of work item linked to. Linked item types can be:
        /// one of: "Epic", "Feature", "Product Backlog Item",
        /// "Bug", or "Task"</param>
        /// <returns>The set of linked items matching the search criteria</returns>

        public IEnumerable<Item> FindLinkedItems
            (Item wi, Func<string, bool> ltPredicate, Func<string, bool> wiTypePredicate)
        {
            return FindLinkedItems(wi.Id, ltPredicate, wiTypePredicate);
        }

        /// <summary>
        /// Recursively find all children and lower descendants of a work
        /// item. Filters on the type of work items found if desired.
        /// </summary>
        /// <param name="wi">The parent work item</param>
        /// <param name="wiTypePredicate">A predicate function that tests
        /// the type of work item linked to. Linked item types can be:
        /// one of: "Epic", "Feature", "Product Backlog Item",
        /// "Bug", or "Task"</param>
        /// <returns>All matching descendants of this work item</returns>
        
        public IEnumerable<Item> FindDescendants(Item wi, Func<string, bool> wiTypePredicate)
        {
            // Find the immediate descendants first

            var allChildren = FindLinkedItems(wi, lt => lt == "Child", null);
            IEnumerable<Item> matchingChildren = allChildren;
            if (wiTypePredicate != null)
                matchingChildren = matchingChildren.Where(i => wiTypePredicate(i.WorkItemType));

            // Include any lower descendants

            foreach (Item i in allChildren)
                matchingChildren = matchingChildren.Concat(FindDescendants(i, wiTypePredicate));
            return matchingChildren;
        }
    }
}
