﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Text.RegularExpressions;

namespace TFSLib
{
    public class SprintManager
    {
        public string Stream { get; private set; }
        public string Sprint { get; private set; }

        public List<ViolationSet> Violations
        {
            get;
            private set;
        }

        public WorkItemAdapter ItemAdapter
        {
            get;
            private set;
        }

        /// <summary>
        /// The list of all streams and other areas
        /// </summary>
        
        public string[] Areas
        {
            get
            {
                return ItemAdapter.AllCurrentWorkItems
                    .Select(i => i.AreaPath)
                    .Where(ap => !ap.Contains("x_Old"))
                    .Distinct()
                    .OrderBy(ap => ap)
                    .ToArray();
            }
        }

        /// <summary>
        /// The list of all sprints or iterations
        /// </summary>
        
        public string[] Iterations
        {
            get
            {
                return ItemAdapter.AllCurrentWorkItems
                    .Select(i => i.IterationPath)
                    .Where(ap => !ap.Contains("x_Old"))
                    .Distinct()
                    .OrderBy(ap => ap)
                    .ToArray();
            }
        }

        /// <summary>
        /// The list of users that have edited items
        /// </summary>
        
        public string[] Editors
        {
            get
            {
                return ItemAdapter.AllCurrentWorkItems
                    .Select(i => i.ChangedBy)
                    .Distinct()
                    .OrderBy(ap => ap)
                    .ToArray();
            }
        }

        public SprintManager(WorkItemAdapter wia, string area, string iter)
        {
            if (wia == null)
                throw new ArgumentException("Null work item adapter passed to SprintManager");
            else
                ItemAdapter = wia;
            if(!area.StartsWith(ItemAdapter.TfsProjectName, StringComparison.CurrentCultureIgnoreCase))
                throw new ArgumentException("Stream name should begin with team project name");
            if (!iter.StartsWith(ItemAdapter.TfsProjectName, StringComparison.CurrentCultureIgnoreCase))
                throw new ArgumentException("Sprint number should begin with team project name");
            Stream = area;
            Sprint = iter;
            Violations = RulesEngine.Run(ItemAdapter);
        }

        private IEnumerable<RuleViolation> getViolationsForItem(int Id)
        {
            var violations = Violations.Find(v => v.Source.Id == Id);
            if (violations != null)
                return violations.Violations;
            else
                return Enumerable.Empty<RuleViolation>();
        }

        public List<EndSprintItem> AllSprintItems
        {
            get
            {
                List<EndSprintItem> allItems = new List<EndSprintItem>();
                foreach (Item i in PBIs.Concat(Bugs))
                {
                    var violations = getViolationsForItem(i.Id);
                    allItems.Add(new EndSprintItem(i, 
                        RecommendationForItem(i, false),
                        ActionsForItem(i),
                        TasksUnder(i).Select(t => t.Effort).Sum(), 
                        TasksUnder(i).Select(t => t.RemainingWork).Sum(),
                        violations));
                    foreach(Item task in TasksUnder(i))
                    {
                        var taskViolations = getViolationsForItem(task.Id);
                        allItems.Add(new EndSprintItem(task,
                            RecommendationForItem(task, false), 
                            ActionsForItem(task), // Always empty array for tasks
                            task.Effort, task.RemainingWork,
                            taskViolations));
                    }
                }
                return allItems;
            }
        }

        /// <summary>
        /// Assess whether an iteration or area
        /// path is under some parent path
        /// </summary>
        /// <param name="root">The parent path</param>
        /// <param name="s">The path being tested</param>
        /// <returns>True is one is under the other</returns>
        
        public static bool Under(string root, string s)
        {
            return s == root || s.StartsWith(root + "\\");
        }

        public IEnumerable<Item> PBIs
        {
            get
            {
                return
                    from wi in ItemAdapter.AllCurrentWorkItems
                    where Under(Sprint, wi.IterationPath)
                        && Under(Stream, wi.AreaPath)
                        && wi.WorkItemType == "Product Backlog Item" 
                    select wi;
            }
        }

        public IEnumerable<Item> Bugs
        {
            get
            {
                return
                    from wi in ItemAdapter.AllCurrentWorkItems
                    where Under(Sprint, wi.IterationPath)
                        && Under(Stream, wi.AreaPath)
                        && wi.WorkItemType == "Bug" 
                    select wi;
            }
        }

        public Item ParentOf(Item i)
        {
            return
                (from l in ItemAdapter.AllCurrentWorkItemLinks
                where l.LinkType == "Child"
                && l.Target.Id == i.Id
                select l.Source)
                .FirstOrDefault();
        }

        public IEnumerable<Item> TasksUnder(Item i)
        {
            return 
                from l in ItemAdapter.AllCurrentWorkItemLinks
                where l.Source.Id == i.Id
                && l.LinkType == "Child"
                && l.Target.WorkItemType == "Task"
                select l.Target;
        }

        /// <summary>
        /// Look up the name of the next sprint by scheduled date in TFS. If
        /// this fails, attempt to guess the next sprint number
        /// as follows: "DataWarehouse\Sprint 63" becomes "DataWarehouse\Sprint 64"
        /// while "DataWarehouse\Sprint 62.1" becomes "DataWarehouse\Sprint 63"
        /// </summary>
        /// <returns>A guess at the next sprint name</returns>
        
        public string SprintAfter()
        {
            ScheduleInfo currSprint = ItemAdapter.SprintSchedules
                .FirstOrDefault(ss=>string.Compare(Sprint, ss.IterationPath, true) == 0);
            if(currSprint != null)
            {
                int currSprintIdx = ItemAdapter.SprintSchedules.IndexOf(currSprint);
                if(currSprintIdx < ItemAdapter.SprintSchedules.Count - 1)
                    return ItemAdapter.SprintSchedules[currSprintIdx + 1].IterationPath;
            }

            string nextSprint = string.Empty; 
            Regex trailingDigits = new Regex("^(" + ItemAdapter.TfsProjectName + ".* )([\\d]+(\\.[\\d]+)?)$");
            Match m = trailingDigits.Match(Sprint);
            if (m.Success)
                nextSprint = m.Groups[1].Value + (int)(Math.Ceiling(double.Parse(m.Groups[2].Value) + .001));
            else
                throw new ArgumentException("Current sprint number badly formed");

            if(!Iterations.Contains(nextSprint))
                throw new InvalidOperationException("A local TFS administrator must create iteration '" 
                    + nextSprint + "' first");

            return nextSprint;
        }

        private string AdvanceState(Item i, string newState)
        {
            i.State = newState;
            return UpdateItem(i);
        }

        private string EmptyCheck(Item i, string ie)
        {
            if (!string.IsNullOrEmpty(ie))
                return "[" + i.Id + "] " + ie + "\r\n";
            else
                return string.Empty;
        }

        private string UpdateItem(Item i)
        {
            return EmptyCheck(i, ItemAdapter.UpdateItem(i));
        }

        private string InsertItem(Item i)
        {
            return EmptyCheck(i, ItemAdapter.InsertItem(i));
        }

        private string InsertLink(Item src, Item tgt, string linkType)
        {
            return EmptyCheck(src, ItemAdapter.InsertLink(src, tgt, linkType));
        }

        private string DeleteLink(Item src, Item tgt, string linkType)
        {
            return EmptyCheck(src, ItemAdapter.DeleteLink(src, tgt, linkType));
        }

        private string AdvanceStateToDone(Item i)
        {
            string err = string.Empty;
            if (i.WorkItemType == "Task")
            {
                if (i.State == "To Do")
                    err += AdvanceState(i, "In Progress");
                if (i.State == "In Progress")
                    err += AdvanceState(i, "Done");
            }
            else if (i.WorkItemType == "Product Backlog Item"
                || i.WorkItemType == "Bug")
            {
                if (i.State == "New")
                    err += AdvanceState(i, "Approved");
                if (i.State == "Approved")
                    err += AdvanceState(i, "Committed");
                if (i.State == "Committed")
                    err += AdvanceState(i, "Done");
            }
            return err;
        }

        public void RollOver(Item itemToBeRolledOver, bool toNextSprint, string nextSprintName)
        {
            // When rolling to the next sprint, either synthesise a default
            // next IterationPath using a standard algorith, or if
            // given a next sprint name as the third argument use that.
            // Should toNextSprint be false, just roll the PBI to the
            // IterationPath that is the same as the project name. This
            // is where the product backlog items are expected to be.

            string nextSprint = ItemAdapter.TfsProjectName;
            if (toNextSprint)
            {
                if (string.IsNullOrEmpty(nextSprintName))
                    nextSprint = SprintAfter();
                else
                    nextSprint = nextSprintName;
            }

            // We only process PBIs and Bugs, as Tasks get processed
            // as part of processing their parents here

            if(itemToBeRolledOver.WorkItemType != "Product Backlog Item" 
                && itemToBeRolledOver.WorkItemType != "Bug")
                throw new ArgumentException("Roll over is applied to PBIs and Bugs only");

            string err = string.Empty;
            List<Item> tasksUnder = TasksUnder(itemToBeRolledOver).ToList();

            // Items that have not been started simply
            // get moved to the next iteration verbatim.
            
            if(itemToBeRolledOver.State != "Done" 
                && (tasksUnder == null 
                || !tasksUnder.Any() 
                || tasksUnder.All(t => t.State == "To Do" 
                    || t.RemainingWork == t.Effort)))
            {
                foreach(Item t in tasksUnder)
                {
                    t.State = "To Do";
                    if (string.IsNullOrEmpty(t.Description))
                        t.Description = "[Rolled from previous sprint]";
                    t.IterationPath = nextSprint;
                    err += UpdateItem(t);
                }
                itemToBeRolledOver.IterationPath = nextSprint;
                err += UpdateItem(itemToBeRolledOver);
                if(!string.IsNullOrEmpty(err))
                    throw new InvalidOperationException(err);
            }

            // Items with child tasks that are part completed
            // need to be split into two, with one PBI remaining
            // in the previous sprint containing any 'Done' tasks,
            // and the other appearing in the next sprint. Tasks
            // with hours remaining or still in the 'To Do' state
            // will be moved to the new PBI in the next sprint.
            // The story points will be split proportionately.

            else if(itemToBeRolledOver.State != "Done" 
                && tasksUnder != null 
                && tasksUnder.Any(t => t.RemainingWork > 0 || t.State == "To Do"))
            {
                double totalHours = TasksUnder(itemToBeRolledOver).Select(t => t.Effort).Sum();
                double remainingHours = TasksUnder(itemToBeRolledOver).Select(t => t.RemainingWork).Sum();
                double rollOverFraction = 1.0;
                if(totalHours < remainingHours)
                    totalHours = remainingHours;
                if(totalHours > 0)
                    rollOverFraction = remainingHours/totalHours;

                // Create the duplicate roll-over story in the next sprint

                Item newItem = new Item
                {
                    AcceptanceCriteria = itemToBeRolledOver.AcceptanceCriteria,
                    AreaPath = itemToBeRolledOver.AreaPath,
                    AssignedTo = itemToBeRolledOver.AssignedTo,
                    ChangedBy = itemToBeRolledOver.ChangedBy,
                    ChangedDate = itemToBeRolledOver.ChangedDate,
                    Description = "[Rolled over] " + itemToBeRolledOver.Description,
                    Effort = Math.Round(itemToBeRolledOver.Effort*rollOverFraction),
                    Id = 0,
                    IterationPath = nextSprint,
                    RemainingWork = itemToBeRolledOver.RemainingWork, // Not relevant to a PBI
                    State = "New",
                    Title = "[Rolled over] " + itemToBeRolledOver.Title,
                    WorkItemType = itemToBeRolledOver.WorkItemType
                };
                err += InsertItem(newItem);

                // Move the state of the PBI forward if it
                // has been selected into the next sprint

                if (toNextSprint)
                {
                    err += AdvanceState(newItem, "Approved");
                    err += AdvanceState(newItem, "Committed");
                }


                // Adjust the story points on the original PBI/Bug

                itemToBeRolledOver.Effort -= newItem.Effort;
                err += UpdateItem(itemToBeRolledOver);
                err += AdvanceStateToDone(itemToBeRolledOver);

                // Duplicate any parent link that was on the original PBI/Bug

                Item feature = ItemAdapter
                    .SelectLinks(itemToBeRolledOver)
                    .Where(l => l.LinkType == "Parent")
                    .Select(l => l.Target)
                    .FirstOrDefault();
                if(feature != null)
                    err += InsertLink(feature, newItem, "Child");

                // Move the child tasks that are not done across to the new sprint

                foreach(Item t in tasksUnder)
                {
                    // Tasks with all hours burned away that have not been
                    // moved to done state get moved to done state automatically

                    if(t.State != "Done" && t.RemainingWork == 0 && t.Effort > 0)
                        err += AdvanceStateToDone(t);

                    // Tasks with remaining hours but partially burned
                    // away are split across the two stories

                    else if(t.RemainingWork < t.Effort && t.RemainingWork > 0)
                    {
                        Item tNew = new Item
                        {
                            AreaPath = t.AreaPath,
                            AssignedTo = t.AssignedTo,
                            ChangedBy = t.ChangedBy,
                            ChangedDate = t.ChangedDate,
                            Description = "[Rolled over] " + t.Description,
                            Effort = t.RemainingWork,
                            Id = 0,
                            IterationPath = nextSprint,
                            RemainingWork = t.RemainingWork,
                            State = "To Do",
                            Title = "[Rolled over] " + t.Title,
                            WorkItemType = t.WorkItemType
                        };
                        err += InsertItem(tNew);

                        // Link the tNew new task to the newItem parent PBI/Bug

                        err += InsertLink(newItem, tNew, "Child");

                        // Close down the half of the task in the preceding sprint

                        t.Effort -= t.RemainingWork;
                        t.RemainingWork = 0;
                        t.Description += " [Rolled into next sprint]";
                        err += AdvanceStateToDone(t);
                    }

                    // Tasks that are unstarted, or where the remaining work is
                    // still the same as the initial estimated effort simply get
                    // moved to the new PBI/Bug in the next sprint

                    else if(t.State == "To Do" || t.RemainingWork == t.Effort )
                    {
                        if (string.IsNullOrEmpty(t.Description))
                            t.Description = "[Rolled from previous sprint]";
                        t.IterationPath = nextSprint;
                        err += UpdateItem(t);

                        // Break the link between the task and its former PBI/Bug

                        err += DeleteLink(itemToBeRolledOver, t, "Child");

                        // Insert a link from the new parent to the task

                        err += InsertLink(newItem, t, "Child");
                    }
                }
                if (!string.IsNullOrEmpty(err))
                    throw new InvalidOperationException(err);
            }

            // All child tasks are done, or have zero hours remaining, but the parent
            // story is still marked as not done. In this case, there must be some
            // reason the parent has not been closed down. This solution marks the
            // story as 'Done' and closes the PBI down.

            else if (itemToBeRolledOver.State != "Done" 
                && TasksUnder(itemToBeRolledOver).All
                (t => t.State == "Done" || t.RemainingWork == 0))
            {
                // Mark the parent as done

                err += AdvanceStateToDone(itemToBeRolledOver);

                // Walk the children, marking any non-done items as done

                foreach (Item t in tasksUnder)
                    err += AdvanceStateToDone(t);

                if (!string.IsNullOrEmpty(err))
                    throw new InvalidOperationException(err);
            }

            // Parent of tasks is marked 'Done'. Ensure all child
            // tasks also moved to 'Done' and close the story down.
            
            else if(itemToBeRolledOver.State == "Done")
            {
                // Walk the children, marking any non-done items as done

                foreach (Item t in tasksUnder)
                    if (t.State != "Done")
                    {
                        t.RemainingWork = 0;
                        err += AdvanceStateToDone(t);
                    }

                if (!string.IsNullOrEmpty(err))
                    throw new InvalidOperationException(err);
            }
        }

        public string RecommendationForItem(Item i, bool identifyItem)
        {
            string identity = string.Empty;
            if (identifyItem)
                identity = string.Format("{0} ({1}) ", i.Id, i.Title);
            if (i.WorkItemType == "Task")
            {
                if (i.State == "To Do")
                    return string.Format("Task{0} will be moved to " +
                        "next sprint or product backlog as part of parent PBI roll over, " +
                        "as it is in 'To Do' state", identity);

                if (i.State == "In Progress"
                    && i.RemainingWork == i.Effort)
                    return string.Format("Task{0} will be moved to" +
                        "next sprint or product backlog as part of parent PBI roll over, " +
                        "as no hours have been consumed this sprint", identity);

                if (i.State == "In Progress"
                    && i.RemainingWork < i.Effort
                    && i.RemainingWork > 0)
                    return string.Format("Task{0} will be split in two, " +
                        "with the remaining effort moved to a new task in the " +
                        "next sprint or product backlog as part of parent PBI roll over, " +
                        "as some hours are remaining in this task", identity);

                if (i.State == "In Progress"
                    && i.RemainingWork == 0)
                    return string.Format("Task{0} will be marked 'Done' " +
                        "and left in the current sprint, as no hours " +
                        "are remaining in this task", identity);

                if (i.State == "Done")
                    return string.Format("Task{0} is 'Done', " +
                        "and will not roll into next sprint", identity);
            }

            if (i.WorkItemType == "Product Backlog Item" || i.WorkItemType == "Bug")
            {
                var tasksUnder = TasksUnder(i);
                if (i.State != "Done"
                    && (tasksUnder == null
                    || !tasksUnder.Any()
                    || tasksUnder.All(t => t.State == "To Do"
                        || t.RemainingWork == t.Effort)))
                    return string.Format(
                        "PBI/Bug{0} is not 'Done' and has no tasks, all unstarted " +
                        "tasks, or all tasks with no hours burned, so it can either be " +
                        "moved to the next sprint along with all its tasks, " +
                        "or moved back to the product backlog", identity);

                if (tasksUnder.Any(t => t.State == "To Do")
                    && tasksUnder.Any(t => t.RemainingWork == 0 || t.State == "Done"))
                {
                    double totalHours = TasksUnder(i).Select(t => t.Effort).Sum();
                    double remainingHours = TasksUnder(i).Select(t => t.RemainingWork).Sum();

                    return string.Format("PBI/Bug{0} has some completed and some unstarted tasks, " +
                        "and should be split in two. " +
                        "The PBI remaining in this sprint will be marked as 'Done' and " +
                        "repointed with {1} points, while the new PBI  will have {2} points " +
                        "and will be placed in the next sprint, or put back on the product backlog. " +
                        "Incomplete tasks will be split in two, " +
                        "and unstarted tasks just moved to the new PBI",
                        identity, (int)(i.Effort * (1 - remainingHours / totalHours)),
                        (int)(i.Effort * remainingHours / totalHours));
                }

                if (tasksUnder.Any(t => t.RemainingWork > 0 || t.State == "To Do"))
                {
                    double totalHours = TasksUnder(i).Select(t => t.Effort).Sum();
                    double remainingHours = TasksUnder(i).Select(t => t.RemainingWork).Sum();

                    return string.Format("PBI/Bug{0} has {3}h work remaining " +
                        "out of {4}h, and should be split in two. " +
                        "The PBI remaining in this sprint will be marked as 'Done' and " +
                        "repointed with {1} points, while the new PBI  will have {2} points " +
                        "and will be placed in the next sprint, or put back on the product backlog. " +
                        "Incomplete tasks will be split in two, " +
                        "and unstarted tasks just moved to the new PBI",
                        identity, (int)(i.Effort * (1 - remainingHours / totalHours)),
                        (int)(i.Effort * remainingHours / totalHours), remainingHours, totalHours);
                }

                if (i.State != "Done" && TasksUnder(i).All(t => t.State == "Done" || t.RemainingWork == 0))
                    return string.Format("PBI/Bug{0} has either all completed " +
                        "tasks, or all tasks with no hours remaining. The " +
                        "PBI/Bug will be marked 'Done' and left in its current sprint", identity);

                if (i.State == "Done")
                    return string.Format("PBI/Bug{0} is marked 'Done' " +
                        "meaning it will not be moved. Any undone child " +
                        "tasks will be forced to 'Done'", identity);
            }

            return string.Format("{0}{1} will not be affected", i.WorkItemType, identity);
        }

        public string[] ActionsForItem(Item i)
        {
            if (i.WorkItemType == "Product Backlog Item" || i.WorkItemType == "Bug")
            {
                var tasksUnder = TasksUnder(i);
                if (i.State != "Done"
                    && (tasksUnder == null
                    || !tasksUnder.Any()
                    || tasksUnder.All(t => t.State == "To Do"
                        || t.RemainingWork == t.Effort)))
                    return new string[] { "Roll over to next sprint", "Put back on product backlog" };

                if (tasksUnder.Any(t => t.State == "To Do")
                    && tasksUnder.Any(t => t.RemainingWork == 0 || t.State == "Done"))
                    return new string[] { "Roll remainder to next sprint", "Put remainder on product backlog" };

                if (tasksUnder.Any(t => t.RemainingWork > 0 || t.State == "To Do"))
                    return new string[] { "Roll remainder to next sprint", "Put remainder on product backlog" };

                if (i.State != "Done" && TasksUnder(i).All(t => t.State == "Done" || t.RemainingWork == 0))
                    return new string[] { "Move to 'Done' state" };

                if (i.State == "Done" && TasksUnder(i).Any(t => t.State != "Done" && t.RemainingWork == 0))
                    return new string[] { "Move undone children to 'Done' state" };
            }

            return new string[0];
        }
    }
}
