﻿/*
 * Author: 
 *  S D Smith
 *  
 * Date:
 *  Oct 2015
 *  
 * Purpose: 
 *  Implementation of the rule checking engine for applying rule checks to TFS.
 *  Used as part of the TFS sanity checks to make sure TFS users in Analytics
 *  conform to recommended best practices in using the work items to
 *  plan and track backlogs and sprints.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;

namespace TFSLib
{
    public static class RulesEngine
    {
        /// <summary>
        /// Retrieve all the loaded assemblies in the curreent applicaiton
        /// </summary>
        /// <returns>The assemblies in the current running application</returns>
        
        public static IEnumerable<Assembly> GetAssemblies()
        {
            //AppDomain currentDomain = AppDomain.CurrentDomain;
            //return currentDomain.GetAssemblies();

            Assembly a = Assembly.GetExecutingAssembly();
            return Enumerable.Repeat(a, 1);
        }

        /// <summary>
        /// Build a list of the types of classes that have
        /// the RuleClass attribute on them
        /// </summary>
        /// <returns>List of the types of classes that have
        /// the RuleClass attribute on them</returns>
        
        public static IEnumerable<Type> GetRuleClasses()
        {
            List<Type> ruleClassTypes = new List<Type>();
            foreach(Assembly a in GetAssemblies())
                foreach(Type t in a.GetTypes())
                    if (t.GetCustomAttribute(typeof(RuleClassAttribute)) != null)
                        ruleClassTypes.Add(t);
            return ruleClassTypes.AsEnumerable();
        }

        /// <summary>
        /// Build a list of instance delegates that point into a
        /// rules validation object and can be used to invoke
        /// each validation rule function in turn.
        /// </summary>
        /// <param name="ruleClassInstance">An instance of a class that
        /// bears the [RuleClass] attribute</param>
        /// <returns>A list of delegates to methods bearing the
        /// [Rule] attribute within the instance of the [RuleClass] object</returns>

        public static IEnumerable<Func<Item, string>> 
            GetWorkItemRules(object ruleClassInstance)
        {
            return findRuleDelegates
                <Func<Item, string>, RuleAttribute>
                    (ruleClassInstance);
        }

        /// <summary>
        /// Build a list of instance delegates that point into a
        /// rules validation object and can be used to invoke
        /// each validation rule function in turn for links.
        /// </summary>
        /// <param name="ruleClassInstance">An instance of a class that
        /// bears the [RuleClass] attribute</param>
        /// <returns>A list of delegates to methods bearing the
        /// [LinkRule] attribute within the instance of the [RuleClass] object</returns>

        public static IEnumerable<Func<Item, string, Item, string>>
            GetLinkRules(object ruleClassInstance)
        {
            return findRuleDelegates
                <Func<Item, string, Item, string>, LinkRuleAttribute>
                    (ruleClassInstance);
        }

        // Shared generic implementation of the delegate builder. Note that the
        // odd 'where T : class' and the use of the 'as T' in CreateDelegate are
        // necessary because generic casts cannot be applied to delegate objects in C#
        
        private static IEnumerable<T>
            findRuleDelegates<T, A>(object ruleClassInstance) where T : class
        {
            // Somewhere to hold the list of delegates to the rule testing functions

            var ruleFuncs = new List<T>();

            // Get the collection of methods on the rules object
            // that was passed in as the parameter to this method

            foreach (MethodInfo mi in ruleClassInstance.GetType().GetMethods())
            {
                // For every method that has the [Rule] attribute on it,
                // create a delegate to that method in the ruleClassInstance,
                // and append the delegate to the list of delegates to be
                // invoked as part of the rules validation process

                if (mi.GetCustomAttribute(typeof(A)) != null)
                    ruleFuncs.Add
                    (
                        Delegate.CreateDelegate
                        (
                            typeof(T), ruleClassInstance, mi
                        ) as T
                    );
            }
            return ruleFuncs;
        }

        /// <summary>
        /// Run the complete set of rule tests against the instance of TFS
        /// </summary>
        /// <param name="tfsProjectName">The name of the team project
        /// </param>
        /// <returns>The complete list of violations found</returns>

        public static List<ViolationSet> Run(string tfsProjectName)
        {
            WorkItemAdapter wia = WorkItemAdapter.ItemAdapter(tfsProjectName);
            return Run(wia);
        }

        /// <summary>
        /// Run the complete set of rule tests against the instance of TFS
        /// </summary>
        /// <param name="wia">A pre-existing work item adapter
        /// </param>
        /// <returns>The complete list of violations found</returns>

        public static List<ViolationSet> Run(WorkItemAdapter wia)
        {
            var violations = new List<ViolationSet>();
            foreach(var ruleClass in GetRuleClasses())
            {
                // Build the set of delegates needed to
                // run all the tests in this rule class

                object ruleClassInstance = Activator
                    .CreateInstance(ruleClass, new object[] { wia });
                var workItemRules =
                    new List<Func<Item, string>>
                            (GetWorkItemRules(ruleClassInstance));
                var linkRules =
                    new List<Func<Item, string, Item, string>>
                            (GetLinkRules(ruleClassInstance));

                // Run the set of rules checks on each work item,
                // then on every link attached to each work item

                foreach (Item source in wia.AllCurrentWorkItems)
                {
                    List<RuleViolation> itemViolations = new List<RuleViolation>();
                    foreach (var workItemRule in workItemRules)
                    {
                        string violation = workItemRule(source);
                        if (violation != null)
                            itemViolations.Add(new RuleViolation(violation));
                    }

                    // Link violation checks are grouped under the source item

                    foreach (ItemLink link in wia.ItemLinksFromSourceId(source.Id))
                        foreach (var linkRule in linkRules)
                        {
                            string linkViolation = linkRule(link.Source, link.LinkType, link.Target);
                            if (linkViolation != null)
                                itemViolations.Add
                                    (new RuleViolation(linkViolation, link.LinkType, link.Target));
                        }
                    if (itemViolations.Count > 0)
                        violations.Add(new ViolationSet(source, itemViolations));
                }
            }
            return violations;
        }
    }
}
