﻿/*
 * Author: 
 *  S D Smith
 *  
 * Date:
 *  Oct 2015
 *  
 * Purpose: 
 *  An entity class representing one violation of a business rule in the
 *  use of TFS.
 *  Used as part of the TFS sanity checks to make sure TFS users in Analytics
 *  conform to recommended best practices in using the work items to
 *  plan and track backlogs and sprints.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace TFSLib
{
    /// <summary>
    /// Container entity for a rule violation that applies
    /// solely to a single work item rather than a link
    /// </summary>

    public class RuleViolation
    {
        /// <summary>
        /// The error message associated with the rule violation
        /// </summary>
        
        public string Message
        {
            get;
            private set;
        }

        /// <summary>
        /// The other work item involved in a violation,
        /// if the rule violation relates to a link
        /// </summary>
        
        public Item Target
        {
            get;
            private set;
        }

        /// <summary>
        /// The type of link between the two work
        /// items that raised a violation
        /// </summary>
        
        public string LinkType
        {
            get;
            private set;
        }

        /// <summary>
        /// Constructor for rule violations involving
        /// a link between two or more work items
        /// </summary>
        /// <param name="msg">The error message describing the violation</param>
        /// <param name="lt">The link type between the source and target work items.
        /// Set to null if the violation does not pertain to a link.</param>
        /// <param name="tgt">The linked work item, if the violation was an inconsistency
        /// related to a link. Null if not a link violation</param>
        
        public RuleViolation(string msg, string lt, Item tgt)
        {
            Message = msg;
            LinkType = lt;
            Target = tgt;
        }

        /// <summary>
        /// Constructor for rule violations
        /// involving a single work item
        /// </summary>
        /// <param name="msg">The error message describing the violation</param>

        public RuleViolation(string msg)
            : this(msg, null, null)
        { }

        /// <summary>
        /// Render the violation in a meaningful way
        /// </summary>
        /// <returns>A description of the rule violation</returns>
        
        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            RenderViolation(sb);
            return sb.ToString();
        }

        /// <summary>
        /// Render the violation message, and the linked
        /// work item if the violation is a link violation
        /// </summary>
        /// <param name="sb">The string builder in which the
        /// violation messages are being reported</param>
        
        public void RenderViolation(StringBuilder sb)
        {
            sb.AppendFormat(" *** {0}", Message);
            sb.AppendLine();
            if (!string.IsNullOrEmpty(LinkType))
            {
                sb.AppendFormat("     Link type = {0}:", LinkType);
                sb.AppendLine();
                RenderWorkItem(sb, Target, 5);
            }
        }

        /// <summary>
        /// Generate a textual description of a work item
        /// </summary>
        /// <param name="sb">The string builder into which we render the item</param>
        /// <param name="wi">The item to be rendered</param>
        
        public static void RenderWorkItem(StringBuilder sb, Item wi, int indent)
        {
            for (int i = 0; i < indent; i++)
                sb.Append(' ');
            sb.AppendFormat("{0} {1}: {2}", wi.Id, wi.WorkItemType, wi.Title);
            sb.AppendLine();
            for (int i = 0; i < indent; i++)
                sb.Append(' ');
            sb.AppendFormat("Area: {0}, Sprint: {1}", wi.AreaPath, wi.IterationPath);
            sb.AppendLine();
            for (int i = 0; i < indent; i++)
                sb.Append(' ');
            sb.AppendFormat("State: {0}, Last changed: {1: dd/MM/yy HH:mm}", 
                wi.State, wi.ChangedDate);
            sb.AppendLine();
        }
    }
}
