﻿/*
 * Author: 
 *  S D Smith
 *  
 * Date:
 *  Oct 2015
 *  
 * Purpose: 
 *  Contains the set of business validation rules applied to TFS when checking
 *  for use that is incompatible with the recommended departmental processes.
 *  Used as part of the TFS sanity checks to make sure TFS users in Analytics
 *  conform to recommended best practices in using the work items to
 *  plan and track backlogs and sprints.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace TFSLib
{
    /// <summary>
    /// Container for the set of business rules being applied
    /// to TFS. Currently hard-coded, but ultimately should
    /// be scripted so that they are configurable.
    /// </summary>
    
    [RuleClass]
    public class RuleValidator
    {
        private WorkItemAdapter workItemAdapter;

        /// <summary>
        /// Constructor. Links the validator to the work item adapter.
        /// </summary>
        /// <param name="wia">The work item adaptor whose work items
        /// and links will be validated</param>
        
        public RuleValidator(WorkItemAdapter wia)
        {
            workItemAdapter = wia;
        }

        /// <summary>
        /// RULE: Epics or features cannot be allocated into a sprint, but should always remain
        /// in the iteration path @ProjectName or @ProjectName\Unplanned
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>
        
        [Rule]
        public string EpicCannotBeInAnIteration(Item wi)
        {
            if ((wi.WorkItemType == "Epic" || wi.WorkItemType == "Feature")
                && wi.IterationPath != workItemAdapter.TfsProjectName
                && wi.IterationPath != workItemAdapter.TfsProjectName + "\\Unplanned")
                return "Epics or features should not be in an iteration";
            else
                return null;
        }

        /// <summary>
        /// RULE: Epics that are no longer in the New state must have an assigned
        /// product owner, lead product owner or an assigned manager to own them.
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string WorkItemMustBeAssignedToAProductOwnerIfNotInNewState(Item wi)
        {
            if ((wi.WorkItemType == "Epic" 
                    || wi.WorkItemType == "Feature" 
                    || wi.WorkItemType == "Product Backlog Item" 
                    || wi.WorkItemType == "Bug")
                && string.IsNullOrEmpty(wi.AssignedTo)
                && wi.State != "New")

                return "Only items in the 'New' state can have no assigned owner";
            else
                return null;
        }

        /// <summary>
        /// RULE: Apart from the defect epic(s), immediate
        /// children of epics must be features or PBIs.
        /// </summary>
        /// <param name="src">The work item being inspected</param>
        /// <param name="linkType">The string description of the link type</param>
        /// <param name="tgt">The work item at the other end of the link</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string ChildrenOfNonDefectEpicMustBeFeaturesOrPBIs
            (Item src, string linkType, Item tgt)
        {
            if (src.WorkItemType == "Epic"
                && !src.Title.Contains("Defect")
                && linkType == "Child"
                && tgt.WorkItemType != "Feature"
                && tgt.WorkItemType != "Product Backlog Item")
                
                return "Children of non-defect epics can only be features or PBIs";
            else
            return null;
        }

        /// <summary>
        /// RULE: Any work item regardless of type should not have an empty description
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string DescriptionCannotBeEmpty(Item wi)
        {
            if (wi.WorkItemType != "Task"
                && wi.WorkItemType != "Bug"
                && wi.State != "Done"
                && wi.State != "Closed"
                && string.IsNullOrEmpty(wi.Description))
                return "Items (except tasks) should not have an empty description";
            else
                return null;
        }

        /// <summary>
        /// RULE: Any bug should not have an empty 'steps to reproduce' field
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string StepsToReproduceCannotBeEmpty(Item wi)
        {
            if (wi.WorkItemType == "Bug"
                && wi.State != "Done"
                && wi.State != "Closed"
                && string.IsNullOrEmpty(wi.Description))
                return "Bugs should not have an empty 'Steps to Reproduce' field";
            else
                return null;
        }

        /// <summary>
        /// RULE: Any work item regardless of type should not have an empty title
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string TitleCannotBeEmpty(Item wi)
        {
            if (string.IsNullOrEmpty(wi.Title))
                return "Items should not have an empty title";
            else
                return null;
        }

        /// <summary>
        /// RULE: Immediate descendants of the defect epic(s)
        /// must be Features or Bugs.
        /// </summary>
        /// <param name="src">The work item being inspected</param>
        /// <param name="linkType">The string description of the link type</param>
        /// <param name="tgt">The work item at the other end of the link</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string ChildrenOfDefectEpicMustBeFeaturesOrBugs
                    (Item src, string linkType, Item tgt)
        {
            if (src.WorkItemType == "Epic"
                && src.Title.Contains("Defect")
                && linkType == "Child"
                && tgt.WorkItemType != "Feature"
                && tgt.WorkItemType != "Bug")

                return "Children of defect epic can only be features or bugs";
            else
                return null;
        }

        /// <summary>
        /// RULE: Descendants of defect epics cannot be regular PBIs
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string DescendantsOfDefectEpicCannotBePBIs(Item wi)
        {
            string message = null;
            var pbiDescendants = workItemAdapter.FindDescendants
                (wi, wiType => wiType == "Product Backlog Item");
            if (wi.WorkItemType == "Epic"
                && wi.Title.Contains("Defect")
                && pbiDescendants.Count() > 0)
            {
                message = "Descendants of defect epic cannot be PBIs (Descendent IDs:";
                foreach(var dwi in pbiDescendants)
                    message += " " + dwi.Id;
                message += ")";
            }
            return  message;
        }

        /// <summary>
        /// RULE: Epics cannot have parents, implying they are at the
        /// top of the hierarchy and cannot have epic children.
        /// </summary>
        /// <param name="src">The work item being inspected</param>
        /// <param name="linkType">The string description of the link type</param>
        /// <param name="tgt">The work item at the other end of the link</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string EpicsShouldHaveNoParents
            (Item src, string linkType, Item tgt)
        {
            if (src.WorkItemType == "Epic"
                && linkType == "Parent")

                return "Epics should have no parent work item";
            else
                return null;
        }

        /// <summary>
        /// RULE: Epics or features with children must not be in the new state
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string EpicsOrFeaturesWithChildrenCannotBeInNewState(Item wi)
        {
            string message = null;

            if ((wi.WorkItemType == "Epic" || wi.WorkItemType == "Feature")
                && wi.State == "New")
            {
                List<ItemLink> links = workItemAdapter.ItemLinksFromSourceId(wi.Id);
                if(links != null && links.Any(il => il.LinkType == "Child"))
                {
                    var children = workItemAdapter
                        .AllCurrentWorkItemLinks
                        .Where(wil => wil.Source.Id == wi.Id && wil.LinkType == "Child");
                    message = "Epics or features with children must not be in 'New' state (Child IDs: ";
                    foreach (var wil in links.Where(l => l.LinkType == "Child"))
                        message += " " + wil.Target.Id;
                    message += ")";
                }
            }
            return message;
        }

        /// <summary>
        /// RULE: Descendants of done epics or features must all be done
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string DescendantsOfDoneEpicsOrFeaturesMustAllBeDone(Item wi)
        {
            string message = null;
            var descendants = workItemAdapter.FindDescendants
                (wi, wiType => true);
            if ((wi.WorkItemType == "Epic" || wi.WorkItemType == "Feature")
                && wi.State == "Done"
                && descendants.Any(d => d.State != "Removed" 
                    && d.State != "Deleted" 
                    && d.State != "Done"))
            {
                message = "An epic or feature can only be 'Done' if all its descendants are 'Done' (Descendent IDs:";
                foreach(var dwi in descendants)
                    message += " " + dwi.Id;
                message += ")";
            }
            return message;
        }

        /// <summary>
        /// RULE: Features may only have a feature or an epic as their parent
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string FeaturesandPBIsMustHaveFeatureOrEpicAsParent
            (Item src, string linkType, Item tgt)
        {
            if ((src.WorkItemType == "Feature" || src.WorkItemType == "Product Backlog Item")
                && linkType == "Parent"
                && tgt.WorkItemType != "Feature"
                && tgt.WorkItemType != "Epic")

                return "Features and PBIs may only have epics or features as parents";
            else
                return null;
        }

        /// <summary>
        /// RULE: Features may only have a feature or PBI or bug as child
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string FeaturesMustHaveFeatureBugOrPBIAsChild
            (Item src, string linkType, Item tgt)
        {
            if ((src.WorkItemType == "Feature")
                && linkType == "Child"
                && tgt.WorkItemType != "Feature"
                && tgt.WorkItemType != "Product Backlog Item"
                && tgt.WorkItemType != "Bug")

                return "Features may only have features, bugs or PBIs as children";
            else
                return null;
        }

        /// <summary>
        /// RULE: Feature areas must match child area if set
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string FeatureOrEpicAreaPathsMustMatchChildIfSet
            (Item src, string linkType, Item tgt)
        {
            if ((src.WorkItemType == "Feature" || src.WorkItemType == "Epic")
                && linkType == "Child"
                && src.AreaPath != workItemAdapter.TfsProjectName
                && src.AreaPath != tgt.AreaPath)

                return "Features' area paths must match their children's, if assigned";
            else
                return null;
        }

        /// <summary>
        /// RULE: Acceptance criteria of non-new PBIs cannot be empty
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string AcceptanceCriteriaOfNonNewPBIsCannotBeEmpty(Item wi)
        {
            if ((wi.WorkItemType == "Product Backlog Item" || wi.WorkItemType == "Bug")
                && wi.State != "New"
                && string.IsNullOrEmpty(wi.AcceptanceCriteria))

                return "PBIs or bugs no longer in 'New' state must have acceptance criteria set";
            else
                return null;
        }

        /// <summary>
        /// RULE: PBIs in New and Approved state cannot be in an iteration
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string PBIsInNewStateMustNotBeInAnIteration(Item wi)
        {
            if ((wi.WorkItemType == "Product Backlog Item" || wi.WorkItemType == "Bug")
                && wi.State == "New"
                && wi.IterationPath != workItemAdapter.TfsProjectName
                && wi.IterationPath != workItemAdapter.TfsProjectName + "\\Unplanned")

                return "PBIs allocated to an iteration should be in the 'Committed' or later state. " +
                    "PBIs in the 'New' state should not be in an iteration";
            else
                return null;
        }

        /// <summary>
        /// RULE: PBIs or bugs not in New and Approved state must be in an iteration
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string PBIsNotInNewAndApprovedStateMustBeInAnIteration(Item wi)
        {
            if ((wi.WorkItemType == "Product Backlog Item" || wi.WorkItemType == "Bug")
                && wi.State != "New" 
                && wi.State != "Approved"
                && (wi.IterationPath == workItemAdapter.TfsProjectName
                    || wi.IterationPath == workItemAdapter.TfsProjectName + "\\Unplanned"))

                return "PBIs or bugs in 'Committed' or a later state should have iteration path set";
            else
                return null;
        }

        /// <summary>
        /// RULE: PBIs not in New state must have been estimated
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string PBIsNotInNewStateMustHaveBeenEstimated(Item wi)
        {
            if ((wi.WorkItemType == "Product Backlog Item" || wi.WorkItemType == "Bug")
                && wi.State != "New"
                && wi.Effort == 0)

                return "PBIs or bugs in 'Approved' or a later state should have estimated effort (points) set";
            else
                return null;
        }

        private bool NullEmptyOrZero(object v)
        {
            return v == null 
                || v is int && ((int)v) == 0 
                || v is float && ((float)v) == 0
                || v is double && ((double)v) == 0
                || string.IsNullOrEmpty(v.ToString());
        }

        /// <summary>
        /// RULE: Descendants of done PBIs or Bugs must all be done
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string DescendantsOfDonePBIsOrBugsMustAllBeDone(Item wi)
        {
            string message = null;
            var descendants = workItemAdapter.FindDescendants
                (wi, wiType => true);
            if ((wi.WorkItemType == "Product Backlog Item" || wi.WorkItemType == "Bug")
                && wi.State == "Done"
                && descendants.Any(d => d.State != "Removed"
                    && d.State != "Deleted"
                    && d.State != "Done"))
            {
                message = "A PBI or bug can only be 'Done' if all its descendants are 'Done' (Descendent IDs:";
                foreach (var dwi in descendants)
                    message += " " + dwi.Id;
                message += ")";
            }
            return message;
        }

        /// <summary>
        /// RULE: Children of PBIs must be tasks or bugs
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string ChildrenOfPBIsMustBeTasksOrBugs
            (Item src, string linkType, Item tgt)
        {
            if (src.WorkItemType == "Product Backlog Item"
                && linkType == "Child"
                && tgt.WorkItemType != "Task"
                && tgt.WorkItemType != "Bug")

                return "Children of PBIs must be tasks or bugs";
            else
                return null;
        }

        /// <summary>
        /// RULE: Children of bugs must be tasks
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string ChildrenOfBugsMustBeTasks
            (Item src, string linkType, Item tgt)
        {
            if (src.WorkItemType == "Bug"
                && linkType == "Child"
                && tgt.WorkItemType != "Task")

                return "Children of bugs must be tasks";
            else
                return null;
        }

        /// <summary>
        /// RULE: Parent of PBIs must be feature or epic
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string ParentOfPBIsMustBeFeatureOrEpic
            (Item src, string linkType, Item tgt)
        {
            if ((src.WorkItemType == "Product Backlog Item")
                && linkType == "Parent"
                && tgt.WorkItemType != "Feature"
                && tgt.WorkItemType != "Epic")

                return "Parent of PBIs must be features or epics";
            else
                return null;
        }

        /// <summary>
        /// RULE: Parent of PBIs must be feature or epic
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string ParentOfBugsMustBeFeaturePBIOrEpic
            (Item src, string linkType, Item tgt)
        {
            if (src.WorkItemType == "Bug"
                && linkType == "Parent"
                && tgt.WorkItemType != "Feature"
                && tgt.WorkItemType != "Epic"
                && tgt.WorkItemType != "Product Backlog Item")

                return "Parent of bugs must be features, PBIs or epics";
            else
                return null;
        }

        /// <summary>
        /// RULE: Only epics can have no parent
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [Rule]
        public string OnlyEpicsCanHaveNoParent(Item wi)
        {
            var parents = workItemAdapter.FindLinkedItems(wi, lt => lt == "Parent", wit => true);
            if (wi.WorkItemType != "Epic" && wi.WorkItemType != "Impediment"
                && (parents == null || parents.Count() == 0))

                return "This work item must have a (not removed) parent, and should ultimately descend from an epic";
            else
                return null;
        }

        /// <summary>
        /// RULE: Parent of tasks must be PBI or Bug
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string ParentOfTasksMustBePBIOrBug
            (Item src, string linkType, Item tgt)
        {
            if (src.WorkItemType == "Task"
                && linkType == "Parent"
                && tgt.WorkItemType != "Bug"
                && tgt.WorkItemType != "Product Backlog Item")

                return "Parents of tasks must be PBIs or bugs";
            else
                return null;
        }

        /// <summary>
        /// RULE: Tasks must be in the same iteration as their parent
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string ParentOfTaskMustBeInSameIteration
            (Item src, string linkType, Item tgt)

        {
            if(src.WorkItemType == "Task"
                && linkType == "Parent"
                && src.IterationPath != tgt.IterationPath)

                return "Tasks must be in the same iteration as their parent";
            else
                return null;
        }

        /// <summary>
        /// RULE: Tasks must be in the same area path as their parent
        /// </summary>
        /// <param name="wi">The work item being inspected</param>
        /// <returns>The violation descriptor for this rule</returns>

        [LinkRule]
        public string ParentOfTaskMustBeInArea
            (Item src, string linkType, Item tgt)
        {
            if (src.WorkItemType == "Task"
                && linkType == "Parent"
                && src.AreaPath != tgt.AreaPath)

                return "Tasks must be assigned to the same area (team) as their parent";
            else
                return null;
        }
    }
}
