﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace TFSLib
{
    public class ReportManager
    {
        private SprintManager sprintManager;
        private string tfsProjectName;

        public ReportManager(string tfsProjName)
        {
            tfsProjectName = tfsProjName;
        }

        public TeamSprintStats CalcStatsFor(string area, string iteration)
        {
            if (!iteration.StartsWith(tfsProjectName + "\\"))
                iteration = tfsProjectName + "\\" + iteration;
            if (!area.StartsWith(tfsProjectName + "\\"))
                area = tfsProjectName + "\\" + area;
            WorkItemAdapter wia = WorkItemAdapter.ItemAdapter(tfsProjectName);
            SprintManager sm = new SprintManager
                (wia, area, iteration);

            TeamSprintStats tss = new TeamSprintStats
            {
                Team = area,
                Sprint = iteration,
                StartingPoints = 0,
                TotalInitialHours = 0,
                TotalHoursBurned = 0
            };

            foreach(Item i in sm.PBIs.Concat(sm.Bugs))
            {
                tss.StartingPoints += i.Effort;
                var tl = sm.TasksUnder(i);
                foreach(Item t in tl)
                {
                    tss.TotalInitialHours += t.Effort;
                    tss.TotalHoursBurned += (t.Effort - t.RemainingWork);
                }
            }

            if (tss.StartingPoints == 0 && tss.TotalInitialHours == 0 && tss.TotalHoursBurned == 0)
                return null;
            else
                return tss;
        }

        /// <summary>
        /// Calculate the velocity and task hour statistics
        /// for all Dev teams across all recent sprints
        /// </summary>
        /// <param name="area">The team for whom we want stats, in
        /// the form "Stream N"</param>
        /// <param name="iteration">The sprint for which we want
        /// stats, in the form "Sprint X"</param>
        /// <returns>A list of statistics for each team-sprint</returns>
        
        public List<TeamSprintStats> CalcStats(string area, string iteration)
        {
            if(!string.IsNullOrWhiteSpace(area) && !area.StartsWith("Stream "))
                return new List<TeamSprintStats>(0);
            if(!string.IsNullOrWhiteSpace(iteration) && !iteration.StartsWith("Sprint "))
                return new List<TeamSprintStats>(0);

            WorkItemAdapter wia = WorkItemAdapter.ItemAdapter(tfsProjectName);
            var teams = wia.Teams;
            var iterations = wia.SprintSchedules;
            var stats = new List<TeamSprintStats>();

            var query = from tn in teams.Keys
                        from it in iterations
                        where (string.IsNullOrWhiteSpace(area) || tn == area)
                        && (string.IsNullOrWhiteSpace(iteration)
                        || it.IterationPath.EndsWith("\\" + iteration))
                        orderby tn, it.IterationPath
                        select new { Team = tn, Sprint = it.IterationPath };
            foreach (var tnit in query)
            {
                var stat = CalcStatsFor(tnit.Team, tnit.Sprint);
                if(stat != null)
                    stats.Add(stat);
            }
            return stats;
        }

        /// <summary>
        /// Given a particular iteration within TFS, generate the per-user
        /// statistics for usage of TFS. Each user has the following data
        /// recorded for them:
        /// Their TFS user name;
        /// Their team name;
        /// The number of times they used TFS to perform an insertion or update;
        /// The average time between updates;
        /// The biggest gap between updates;
        /// The last occasion they performed an update.
        /// </summary>
        /// <param name="iteration">The sprint for which statistics are required</param>
        /// <returns>The complete usage report for the selected sprint</returns>
        
        public List<TFSUsageGroup> TfsUsageReport(string iteration)
        {
            if (!iteration.StartsWith(tfsProjectName + "\\"))
                iteration = tfsProjectName + "\\" + iteration;
            WorkItemAdapter wia = WorkItemAdapter.ItemAdapter(tfsProjectName);
            var teams = wia.Teams;
            var usageReport = new List<TFSUsageGroup>();
            foreach(string teamName in teams.Keys
                .Where(k=>k.Contains("Stream ") || k.Contains("Product "))
                .OrderBy(k=>k))
            {
                SprintManager sm = new SprintManager
                    (wia, tfsProjectName + "\\" + teamName,  iteration);
                TFSUsageGroup usageGroup = new TFSUsageGroup(teamName);
                List<string> teamMembers = teams[teamName];
                
                foreach(string teamMember in teamMembers)
                {
                    TFSUsage tfsUsage = new TFSUsage
                    {
                        Team = teamName,
                        User = teamMember
                    };

                    var updateTimes = new List<DateTime>();
                    foreach(Item i in sm.PBIs.Concat(sm.Bugs))
                    {
                        wia.PopulateHistory(i);
                        updateTimes.AddRange(GetUpdatesBy(i, teamMember));
                        var tl = sm.TasksUnder(i);
                        foreach(Item t in tl)
                        {
                            wia.PopulateHistory(t);
                            updateTimes.AddRange(GetUpdatesBy(t, teamMember));
                        }
                    }
                    tfsUsage.Updates = updateTimes.Count;
                    if(updateTimes.Count > 0)
                        tfsUsage.LastUpdate = updateTimes.Max();
                    tfsUsage.MaxInterval = MaxInterval(updateTimes);
                    tfsUsage.AverageFrequency = AverageInterval(updateTimes);
                    usageGroup.Members.Add(tfsUsage);
                }
                usageReport.Add(usageGroup);
            }
            return usageReport;
        }

        private static string MaxInterval(List<DateTime> times)
        {
            TimeSpan maxInterval = new TimeSpan(0L);
            if (times.Count > 1)
            {
                for (int i = 0; i < times.Count - 1; i++)
                {
                    TimeSpan interval = times[i + 1] - times[i];
                    if (interval > maxInterval)
                        maxInterval = interval;
                }
                return maxInterval.ToString("dd'd 'hh'h 'mm'm'");
            }
            else
                return "-";
        }

        private static string AverageInterval(List<DateTime> times)
        {
            List<TimeSpan> intervals = new List<TimeSpan>();
            if (times.Count > 1)
            {
                for (int i = 0; i < times.Count - 1; i++)
                    intervals.Add(times[i + 1] - times[i]);
                return new TimeSpan((long)(intervals.Select(i => i.Ticks).Average() + 0.5))
                    .ToString("dd'd 'hh'h 'mm'm'");
            }
            else
            {
                return "-";
            }
        }

        private static IEnumerable<DateTime> GetUpdatesBy(Item i, string who)
        {
            return i.History.Where(item => item.ChangedBy == who).Select(item => item.ChangedDate);
        }
    }
}
