﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Drawing;

namespace ImageCharts
{
    /// <summary>
    /// A set of points to be plotted
    /// </summary>

    public class PointCollection : IList<PointF>
    {
        private List<PointF> points;
        private List<string> barLabels;

        public PointCollection()
            :this(Enumerable.Empty<PointF>())
        { }

        public PointCollection(IEnumerable<PointF> ep)
        {
            points = new List<PointF>(ep);
            barLabels = null;
            LineColour = Color.Empty;
            FillColour = Color.Empty;
        }

        public Color LineColour
        {
            get;
            set;
        }

        public Color FillColour
        {
            get;
            set;
        }

        public string BarLabel(int index)
        {
            if (!HasBarLabels || index < 0 || index >= Count)
                return null;
            return barLabels[index];
        }

        public void AddBar(string label, float value)
        {
            if (barLabels == null)
                barLabels = new List<string>();
            Add(new PointF(10f + barLabels.Count, value));
            barLabels.Add(label);
        }

        public bool HasBarLabels
        {
            get
            {
                return barLabels != null;
            }
        }

        public bool MatchingBarLabels(PointCollection ps)
        {
            if (ps == null)
                throw new ArgumentException("Null point collection");
            if (HasBarLabels != ps.HasBarLabels)
                return false;
            if (HasBarLabels)
                return barLabels.Zip(ps.barLabels, 
                    (s1, s2) => s1 == s2).All(beq => beq);
            return true;
        }

        public string XAxis
        {
            get;
            set;
        }

        public string YAxis
        {
            get;
            set;
        }

        public string Description
        {
            get;
            set;
        }

        public int Count
        {
            get
            {
                return ((IList<PointF>)points).Count;
            }
        }

        public bool IsReadOnly
        {
            get
            {
                return ((IList<PointF>)points).IsReadOnly;
            }
        }

        public PointF this[int index]
        {
            get
            {
                return ((IList<PointF>)points)[index];
            }

            set
            {
                ((IList<PointF>)points)[index] = value;
            }
        }

        public int IndexOf(PointF item)
        {
            return ((IList<PointF>)points).IndexOf(item);
        }

        public void Insert(int index, PointF item)
        {
            ((IList<PointF>)points).Insert(index, item);
        }

        public void RemoveAt(int index)
        {
            ((IList<PointF>)points).RemoveAt(index);
        }

        public void Add(PointF item)
        {
            ((IList<PointF>)points).Add(item);
        }

        public void Clear()
        {
            ((IList<PointF>)points).Clear();
        }

        public bool Contains(PointF item)
        {
            return ((IList<PointF>)points).Contains(item);
        }

        public void CopyTo(PointF[] array, int arrayIndex)
        {
            ((IList<PointF>)points).CopyTo(array, arrayIndex);
        }

        public bool Remove(PointF item)
        {
            return ((IList<PointF>)points).Remove(item);
        }

        public IEnumerator<PointF> GetEnumerator()
        {
            return ((IList<PointF>)points).GetEnumerator();
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return ((IList<PointF>)points).GetEnumerator();
        }
    }
}
