﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Drawing;
using System.Drawing.Drawing2D;

namespace ImageCharts
{
    public enum PlotType
    {
        Points,
        Crosses,
        Lines,
        Bars
    }

    [Flags]
    public enum TextAlignments
    {
        HMiddle = 1,
        Right = 2,
        VMiddle = 4,
        Bottom = 8
    }

    public class Plot
    {
        private Size plotSize;
        private List<PointCollection> curves;

        public PlotType CurveType
        {
            get;
            set;
        }

        public Plot(int x, int y)
        {
            if (x <= 0 || y <= 0)
                throw new ArgumentException("Plot bitmap must have a real size");

            CurveType = PlotType.Lines;
            ForceXAxis = true;
            ForceYAxis = true;

            plotSize = new Size(x, y);
            curves = new List<PointCollection>();
        }

        public void AddPointSet(PointCollection ps)
        {
            if (curves.Count > 0 && (ps.XAxis != curves[0].XAxis || ps.YAxis != curves[0].YAxis))
                throw new ArgumentException("Multiple curves on a plot must have same X and Y axis labels");
            if (curves.Any(cps => !cps.MatchingBarLabels(ps)))
                throw new ArgumentException("X axis labels for bar charts must match exactly for multiiple plots");
            curves.Add(ps);
        }

        public string BarLabel(int index)
        {
            if (curves != null && curves.Count > 0 && curves[0].HasBarLabels)
                foreach (PointCollection ps in curves)
                    if (ps.BarLabel(index) != null)
                        return ps.BarLabel(index);
            return null;
        }

        public bool ForceXAxis { get; set; }
        public bool ForceYAxis { get; set; }


        private RectangleF CalcMaxMin(PlotType pt)
        {
            if (curves.Count <= 0 || curves[0].Count <= 0)
                throw new InvalidOperationException("Plotting pointless curve(s)");

            float
                xMin = float.MaxValue,
                xMax = float.MinValue,
                yMin = float.MaxValue,
                yMax = float.MinValue,
                xLeftMargin = 0,
                xRightMargin = 0;

            foreach (PointCollection ps in curves)
            {
                foreach (PointF p in ps)
                {
                    if (p.X < xMin)
                        xMin = p.X;
                    if (p.Y < yMin)
                        yMin = p.Y;
                    if (p.X > xMax)
                        xMax = p.X;
                    if (p.Y > yMax)
                        yMax = p.Y;
                }
                if(ps.Count > 1 && ps[0].X == xMin && 2*xLeftMargin < ps[1].X - ps[0].X)
                    xLeftMargin = (ps[1].X - ps[0].X) / 2;
                if (ps.Count > 1 && ps[ps.Count - 1].X == xMax && 2 * xRightMargin < ps[ps.Count - 1].X - ps[ps.Count - 2].X)
                    xRightMargin = (ps[ps.Count - 1].X - ps[ps.Count - 2].X);
            }
            
            if(pt == PlotType.Bars)
            {
                xMin -= xLeftMargin;
                xMax += xRightMargin;
            }
            else
            {
                xMin -= (xMax - xMin) / 20;
                xMax += (xMax - xMin) / 20;
                yMin -= (yMax - yMin) / 20;
                yMax += (yMax - yMin) / 20;
            }

            if (ForceXAxis)
            {
                if (xMax < 0)
                    xMax = 0;
                else if (xMin > 0)
                    xMin = 0;
            }
            if (ForceYAxis)
            {
                if (yMax < 0)
                    yMax = 0;
                else if (yMin > 0)
                    yMin = 0;
            }

            return new RectangleF(xMin, yMin, xMax - xMin, yMax - yMin);
        }

        private float scaleX;
        private float scaleY;
        private RectangleF bounds;

        private static void SetHighQualityRendering(Graphics g)
        {
                g.CompositingMode = CompositingMode.SourceOver;
                g.CompositingQuality = CompositingQuality.HighQuality;
                g.InterpolationMode = InterpolationMode.HighQualityBicubic;
                g.SmoothingMode = SmoothingMode.HighQuality;
                g.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;
                g.PixelOffsetMode = PixelOffsetMode.HighQuality;
        }

        public Image Render()
        {
            Bitmap bmp = new Bitmap(plotSize.Width, plotSize.Height);
            using (Graphics g = Graphics.FromImage(bmp))
            {
                SetHighQualityRendering(g);

                // Paint the background white

                g.Clear(Color.White);

                float pixKeyHeight = RenderKey(bmp, g);

                bounds = CalcMaxMin(CurveType);

                scaleX = bounds.Width / plotSize.Width;
                scaleY = bounds.Height / (plotSize.Height - pixKeyHeight);
                float dx = UnitSize(bounds.Width);
                float dy = UnitSize(bounds.Height);

                // Each pixel on the X axis corresponds to
                // bounds.Width/plotSize.Width. Similarly
                // for the Y axis. The origin has to be translated
                // so that the minimum X and Y values appear at the
                // edge of the plot.

                g.ScaleTransform((float)(1 / scaleX), -(float)(1 / scaleY));
                g.TranslateTransform((float)(-bounds.Left), (float)(-bounds.Bottom));

                // Draw the vertical grid lines and Y axis

                DrawGridLines(bmp, g, dx, dy);

                // Now render the graph content

                for (int i = 0; i < curves.Count; i++)
                    RenderPoints(bmp, g, i);

                // Render the axes labels over the top

                DrawAxesValues(bmp, g, dx, dy);
            }
            return bmp;
        }

        private void DrawAxesValues(Bitmap bmp, Graphics g, float dx, float dy)
        {
            // Draw the barchart labels or axis values along the X axis

            int index = 0;
            for (double v = GridCeil(bounds.Left, dx); v <= GridFloor(bounds.Right, dx); v += dx)
            {
                if (!curves[0].HasBarLabels)
                {
                    string label = string.Format("{0:G}", v);
                    RenderString(bmp, g, label,
                        new RectangleF((float)v, (float)(bounds.Top), dx, -bounds.Height),
                        TextAlignments.HMiddle | TextAlignments.Bottom);
                }
                else
                {
                    RenderString(bmp, g, BarLabel(index), new RectangleF
                        (10f + index, bounds.Top, 1f, -bounds.Height),
                        TextAlignments.HMiddle | TextAlignments.Bottom);
                    index++;
                }
            }

            // Draw the Y axis labels

            for (double v = GridCeil(bounds.Top, dy); v <= GridFloor(bounds.Bottom, dy); v += dy)
            {
                string label = string.Format("{0:G}", v);
                RenderString(bmp, g, label,
                    new RectangleF((float)(bounds.Left), (float)v, bounds.Width, -bounds.Height),
                    TextAlignments.VMiddle);
            }
        }
        private void DrawGridLines(Image bmp, Graphics g, float dx, float dy)
        {
            for (double v = GridCeil(bounds.Left, dx); v <= GridFloor(bounds.Right, dx); v += dx)
                RenderGridLine(bmp, g, v==0, v, bounds.Top, v, bounds.Bottom);

            // Draw the horizontal grid lines and X axis

            for (double v = GridCeil(bounds.Top, dy); v <= GridFloor(bounds.Bottom, dy); v += dy)
                RenderGridLine(bmp, g, v == 0, bounds.Left, v, bounds.Right, v);
        }

        private void RenderPoints(Image bmp, Graphics g, int i)
        {
            PointCollection ps = curves[i];
            using (Pen p = new Pen(ps.LineColour, (float)scaleX))
            {
                using (Brush b = new SolidBrush(ps.FillColour))
                {
                    if (CurveType == PlotType.Lines)
                    {
                        PointF[] points = new PointF[ps.Count + 2];
                        for (int idx = 0; idx < ps.Count; idx++)
                            points[idx] = new PointF(ps[idx].X, ps[idx].Y);
                        points[points.Length - 2] = new PointF(ps[ps.Count - 1].X, 0f);
                        points[points.Length - 1] = new PointF(ps[0].X, 0f);
                        if (ps.FillColour != Color.Empty)
                            g.FillPolygon(b, points);
                        if (ps.LineColour != Color.Empty)
                        {
                            PointF[] linePoints = points.Take(points.Length - 2).ToArray();
                            g.DrawLines(p, linePoints);
                        }
                    }
                    else if (CurveType == PlotType.Bars)
                    {
                        for (int idx = 0; idx < ps.Count; idx++)
                        {
                            float xHalfWidth;
                            if (idx == 0 && ps.Count > 1)
                                xHalfWidth = (ps[1].X - ps[0].X) / 3;
                            else if (idx > 0)
                                xHalfWidth = (ps[idx].X - ps[idx - 1].X) / 3;
                            else
                                xHalfWidth = 1.0f;

                            var points = new PointF[]
                            {
                        new PointF(ps[idx].X - xHalfWidth, 0f),
                        new PointF(ps[idx].X - xHalfWidth, ps[idx].Y),
                        new PointF(ps[idx].X+xHalfWidth,ps[idx].Y),
                        new PointF(ps[idx].X+xHalfWidth, 0f),
                            };
                            if (ps.FillColour != Color.Empty)
                                g.FillPolygon(b, points);
                            if (ps.LineColour != Color.Empty)
                                g.DrawLines(p, points);
                        }
                    }
                    else if (CurveType == PlotType.Crosses || CurveType == PlotType.Points)
                    {

                        for (int idx = 0; idx < ps.Count; idx++)
                        {
                            RenderPoint(bmp, g, string.Empty, ps[idx],
                                TextAlignments.Bottom | TextAlignments.Right, ps.LineColour,
                                CurveType == PlotType.Crosses);
                        }
                    }
                }
            }
        }

        private static void RenderGridLine
            (Image bmp, Graphics gp, bool onAxis, double x1, double y1, double x2, double y2)
        {
            PointF[] destPt = new PointF[] 
                { new PointF((float)x1, (float)y1), new PointF((float)x2, (float)y2)};
            gp.TransformPoints(CoordinateSpace.Page, CoordinateSpace.World, destPt);
            using (Graphics g = Graphics.FromImage(bmp))
            {
                SetHighQualityRendering(g);
                g.ResetTransform();
                using (Pen p = new Pen(Color.LightGray, onAxis ? 3 : 1))
                    g.DrawLine(p, destPt[0], destPt[1]);
            }
        }

        private static void RenderPoint(Image bmp, Graphics gp, string s, 
            PointF where, TextAlignments ta, Color penColour, bool useCrosses)
        {
            PointF[] destPt = new PointF[] { where };
            gp.TransformPoints(CoordinateSpace.Page, CoordinateSpace.World, destPt);
            using (Graphics g = Graphics.FromImage(bmp))
            {
                SetHighQualityRendering(g);
                g.ResetTransform();

                // Compute size of cross

                float crossDiameter = bmp.Size.Width;
                if (bmp.Size.Height > crossDiameter)
                    crossDiameter = bmp.Size.Height;
                crossDiameter /= 20;
                if (crossDiameter > 16)
                    crossDiameter = 16;

                // Draw the cross

                if (useCrosses)
                {
                    using (Pen p = new Pen(penColour))
                    {
                        g.DrawLine(p, destPt[0].X - crossDiameter / 2, destPt[0].Y,
                            destPt[0].X + crossDiameter / 2, destPt[0].Y);
                        g.DrawLine(p, destPt[0].X, destPt[0].Y - crossDiameter / 2,
                            destPt[0].X, destPt[0].Y + crossDiameter / 2);
                    }
                }
                else
                {
                    using (Brush b = new SolidBrush(penColour))
                        g.FillEllipse(b, destPt[0].X - 2, destPt[0].Y - 2, 5f, 5f);
                }

                // Label the point if a label required

                if (!string.IsNullOrEmpty(s))
                {
                    using (Font f = new Font("Arial", 10f))
                    {
                        SizeF txtSize = g.MeasureString(s, f);
                        if ((ta & TextAlignments.HMiddle) != 0)
                            destPt[0].X -= txtSize.Width / 2;
                        else if ((ta & TextAlignments.Right) != 0)
                            destPt[0].X -= txtSize.Width;
                        if ((ta & TextAlignments.VMiddle) != 0)
                            destPt[0].Y -= txtSize.Height / 2;
                        else if ((ta & TextAlignments.Bottom) != 0)
                            destPt[0].Y -= txtSize.Height;
                        g.DrawString(s, f, Brushes.Black, destPt[0]);
                    }
                }
            }
        }

        private float RenderKey(Image bmp, Graphics g)
        {
            if (bmp.Width < 120)
                return 0;

            // Measure the key rectangle

            PointF[] txtLocations = new PointF[curves.Count + 2];

            PointF nextEvenLoc = new PointF(24f, 0f);
            PointF nextOddLoc = new PointF(bmp.Width / 2 + 24f, 0f);
            using (Font f = new Font("Arial", 10f))
            {
                if (!string.IsNullOrEmpty(curves[0].XAxis))
                {
                    SizeF txtSize = g.MeasureString
                        ("Horizontal axis: " + curves[0].XAxis, f, bmp.Width / 2);
                    txtLocations[0] = nextEvenLoc;
                    nextEvenLoc.Y += txtSize.Height;
                }
                if (!string.IsNullOrEmpty(curves[0].YAxis))
                {
                    SizeF txtSize = g.MeasureString
                        ("Vertical axis: " + curves[0].YAxis, f, bmp.Width / 2);
                    txtLocations[1] = nextOddLoc;
                    nextOddLoc.Y += txtSize.Height;
                }
                for (int i = 0; i < curves.Count; i += 2)
                {
                    SizeF txtSize = g.MeasureString
                        (curves[i].Description, f, bmp.Width / 2 - 24);
                    txtLocations[i + 2] = nextEvenLoc;
                    nextEvenLoc.Y += txtSize.Height;
                }
                for (int i = 1; i < curves.Count; i += 2)
                {
                    SizeF txtSize = g.MeasureString
                        (curves[i].Description, f, bmp.Width / 2 - 24);
                    txtLocations[i + 2] = nextOddLoc;
                    nextOddLoc.Y += txtSize.Height;
                }
            }
            float height = nextEvenLoc.Y;
            if (nextOddLoc.Y > height)
                height = nextOddLoc.Y;
            if (height+12 >= bmp.Height)
                return 0;

            for (int i = 2; i < txtLocations.Length; i++)
            {
                txtLocations[i].Y += bmp.Height - height;
                RenderString(bmp, g, curves[i-2].Description, new RectangleF
                    (txtLocations[i].X, txtLocations[i].Y, bmp.Width/2 - 24, bmp.Height), 0);
                using (Pen p = new Pen(curves[i - 2].LineColour))
                using (Brush b = new SolidBrush(curves[i - 2].FillColour))
                using (Brush pointBrush = new SolidBrush(curves[i - 2].LineColour))
                {
                    if (CurveType == PlotType.Lines || CurveType == PlotType.Bars)
                    {
                        g.FillRectangle(b, txtLocations[i].X - 18,
                            txtLocations[i].Y + 2, 12f, 12f);
                        g.DrawRectangle(p, txtLocations[i].X - 18,
                            txtLocations[i].Y + 2, 12f, 12f);
                    }
                    else if (CurveType == PlotType.Points)
                        g.FillEllipse(pointBrush, txtLocations[i].X - 18,
                            txtLocations[i].Y + 2, 12f, 12f);
                    else if (CurveType == PlotType.Crosses)
                    { 
                        g.DrawLine(p, txtLocations[i].X - 12, txtLocations[i].Y + 2, 
                            txtLocations[i].X - 12, txtLocations[i].Y + 14);
                        g.DrawLine(p, txtLocations[i].X - 18, txtLocations[i].Y + 8, 
                            txtLocations[i].X - 6, txtLocations[i].Y + 8);
                    }
                }
            }
            if (!string.IsNullOrEmpty(curves[0].XAxis))
            {
                txtLocations[0].Y += bmp.Height - height;
                txtLocations[0].X -= 18;
                RenderString(bmp, g, "Horizontal axis: " + curves[0].XAxis, new RectangleF
                    (txtLocations[0].X,
                    txtLocations[0].Y, bmp.Width / 2, bmp.Height), 0);
            }
            if (!string.IsNullOrEmpty(curves[0].YAxis))
            {
                txtLocations[1].Y += bmp.Height - height;
                txtLocations[1].X -= 18;
                RenderString(bmp, g, "Vertical axis: " + curves[0].YAxis, new RectangleF
                    (txtLocations[1].X,
                    txtLocations[1].Y, bmp.Width / 2, bmp.Height), 0);
            }

            return height + 12;
        }

        private static void RenderString(Image bmp, Graphics gp, string s, RectangleF where, TextAlignments ta)
        {
            PointF[] destPt = new PointF[] { where.Location, new PointF(where.Right, where.Bottom) };
            gp.TransformPoints(CoordinateSpace.Page, CoordinateSpace.World, destPt);
            using (Graphics g = Graphics.FromImage(bmp))
            {
                SetHighQualityRendering(g);
                g.ResetTransform();
                using (Font f = new Font("Arial", 10f))
                {
                    SizeF txtSize = g.MeasureString(s, f);
                    if ((ta & TextAlignments.HMiddle) != 0)
                        destPt[0].X -= txtSize.Width / 2;
                    else if ((ta & TextAlignments.Right) != 0)
                        destPt[0].X -= txtSize.Width;
                    if ((ta & TextAlignments.VMiddle) != 0)
                        destPt[0].Y -= txtSize.Height / 2;
                    else if ((ta & TextAlignments.Bottom) != 0)
                        destPt[0].Y -= txtSize.Height;
                    g.DrawString(s, f, Brushes.Black, new RectangleF(destPt[0].X, destPt[0].Y,
                        destPt[1].X - destPt[0].X, destPt[1].Y - destPt[0].Y));
                }
            }
        }

        private static double GridFloor(double value, double gridSpacing)
        {
            return Math.Floor(value / gridSpacing) * gridSpacing;
        }

        private static double GridCeil(double value, double gridSpacing)
        {
            return gridSpacing + GridFloor(value, gridSpacing);
        }

        private static float UnitSize(double range)
        {
            int pow = 0;
            while (Math.Pow(10.0, pow) < range)
                pow++;
            do
            {
                pow--;
            } while (Math.Pow(10.0, pow) > range);

            double unitSize = Math.Pow(10.0, pow);
            if (5 * unitSize < range) // 5 .. 10 steps of 1.0
                return (float)unitSize;
            if (2.5 * unitSize < range) // 2.5 to 5 steps of 0.5
                return (float)(unitSize * 0.5);
            if (2.0 * unitSize < range) // 2 to 2.5 steps of 0.25
                return (float)(unitSize * 0.25);
            return (float)(unitSize * 0.2); // 1 to 2 steps of 0.2
        }
    }
}
