﻿using System;
using GoogleMapClient;

namespace CourierManager
{
    /// <summary>
    /// Record of a booking of a courier to ship a parcel between two addresses
    /// </summary>

    public class CourierBooking
    {

        public static readonly TimeSpan PickupWaitingTime = TimeSpan.FromMinutes(5.0);
        public static readonly TimeSpan DropOffWaitingTime = TimeSpan.FromMinutes(5.0);

        private IFareCalculator journeyPricer;
        private IPremiumCalculator premiumCalculator;

        public long Id { get; private set; }
        public IGoogleRoute Route { get; private set; }
        public double Weight { get; private set; }
        public string ShippingRef { get; set; }
        public string Origin => Route.Origin;
        public string Destination => Route.Destination;
        public double Distance => Route.Distance;

        public DateTime EstimatedPickupTime { get; set; }
        public DateTime EstimatedDropOffTime => EstimatedPickupTime + PickupWaitingTime + Route.Duration;

        // Route courier takes to get to pickup location.
        // Determined once a courier has been assigned

        public IGoogleRoute PickupRoute { get; set; }

        // Used for premium pricing courier service

        public HazardType PackageHazardType { get; set; }
        public int PackageValue { get; set; }


        public CourierBooking(double weight, IGoogleRoute route, IFareCalculator fc, IPremiumCalculator pc = null)
        {
            Id = IdentityAllocator.Instance.IdentityOf(this);
            journeyPricer = fc;
            premiumCalculator = pc;

            Weight = weight;
            Route = route;

            // Used for premiumm pricing

            PackageValue = 0;
            PackageHazardType = HazardType.NonHazardous;
        }

        /// <summary>
        /// Calculate the fare to be charged for delivering this package
        /// </summary>

        public int Fare
        {
            get
            {
                int fare = journeyPricer.CalculateFare(Distance, Weight);
                if (premiumCalculator != null)
                    fare += premiumCalculator.CalculatePremium(PackageHazardType, PackageValue, Distance, Weight);
                return fare;
            }
        }
    }
}
