﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Linq;
using GoogleMapClient;

namespace CourierManager
{
    /// <summary>
    /// Describe a courier and his/her workload
    /// </summary>

    public class Courier
    {
        private IGoogleMapApi mapApi;

        /// <summary>
        /// The sequence of bookings this courier has to work through
        /// </summary>

        public IList<CourierBooking> Schedule { get; private set; } 
            = new List<CourierBooking>();

        public string Name { get; private set; }
        public string InitialLocation { get; private set; }
        public DateTime TimeAtLocation { get; private set; }

        public string LocationWhenNextFree
        {
            get
            {
                if (Schedule == null || Schedule.Count <= 0)
                    return InitialLocation;
                else
                    return Schedule.Last().Destination;
            }
        }

        public DateTime TimeWhenNextFree
            => Schedule == null || Schedule.Count == 0
            ? DateTime.Now
            : Schedule.Last().EstimatedDropOffTime + CourierBooking.DropOffWaitingTime;

        public Courier(IGoogleMapApi gma, string name, string location)
        {
            mapApi = gma;
            Name = name;
            InitialLocation = location;
            TimeAtLocation = DateTime.Now;
        }

        public DateTime CouldArriveAt(IGoogleMapApi mapApi, string location)
        {
            if (string.Compare(LocationWhenNextFree, location, true) == 0)
                return TimeWhenNextFree;

            IGoogleRoute route = mapApi.GetRoute(LocationWhenNextFree, location);
            if (route.Status == "OK")
                return TimeWhenNextFree + route.Duration;
            else
                throw new ArgumentException("Route not found: " + route.Status);
        }

        public void TakeBooking(CourierBooking booking)
        {
            IGoogleRoute pickupRoute = mapApi.GetRoute(LocationWhenNextFree, booking.Route.Origin);
            if (pickupRoute.Status != "OK")
                throw new ArgumentException("Courier unable to take assigned booking");
            booking.EstimatedPickupTime = TimeWhenNextFree + pickupRoute.Duration;
            booking.PickupRoute = pickupRoute;
            Schedule.Add(booking);
        }

        public override string ToString()
        {
            return $"{LocationWhenNextFree} @ {TimeWhenNextFree:t}";
        }
    }
}
