﻿using System;
using System.Net.Http;
using System.Threading.Tasks;
using Newtonsoft.Json;

namespace GoogleMapClient
{
    // Data structures that mirror the returned JSON from
    // Google's mapping API

    internal class JsonMapResponse
    {
        public string status { get; set; }
        public string error_message { get; set; }
        public string[] origin_addresses { get; set; }
        public string[] destination_addresses { get; set; }
        public JsonMapRow[] rows { get; set; }
    }

    internal class JsonMapRow
    {
        public JsonMapElement[] elements { get; set; }
    }

    internal class JsonMapElement
    {
        public string status { get; set; }
        public JsonTextValPair distance { get; set; }
        public JsonTextValPair duration { get; set; }
    }

    internal class JsonTextValPair
    {
        public string text { get; set; }
        public double value { get; set; }
    }

    /// <summary>
    /// Implementation of the interface to the Google mapping API
    /// to measure distances and durations of journeys
    /// </summary>

    public class DirectionsApi : IGoogleMapApi
    {
        private string apiKey;

        public DirectionsApi(string key)
        {
            apiKey = key;
        }

        public IGoogleRoute GetRoute(string origin, string destination)
        {
            var task = GetRouteAsync(origin, destination);
            return task.Result;
        }

        public async Task<IGoogleRoute> GetRouteAsync(string origin, string destination)
        {
            string url = CreateRequestUrl(origin, destination);
            using (var client = new HttpClient())
            {
                using (var response = await client.GetAsync(url))
                {
                    using (var content = response.Content)
                    {
                        string result = await content.ReadAsStringAsync();
                        JsonMapResponse jmr = JsonConvert.DeserializeObject<JsonMapResponse>(result);
                        if (jmr != null
                            && jmr.status != null
                            && jmr.status == "OK"
                            && jmr.origin_addresses != null
                            && jmr.origin_addresses.Length > 0
                            && jmr.destination_addresses != null
                            && jmr.destination_addresses.Length > 0
                            && jmr.rows != null
                            && jmr.rows.Length > 0
                            && jmr.rows[0].elements != null
                            && jmr.rows[0].elements.Length > 0
                            && jmr.rows[0].elements[0].status == "OK")
                            return new GoogleRoute
                            {
                                Status = jmr.status,
                                Origin = jmr.origin_addresses[0],
                                Destination = jmr.destination_addresses[0],
                                Distance = jmr.rows[0].elements[0].distance.value / 1000,
                                Duration = TimeSpan.FromSeconds(jmr.rows[0].elements[0].duration.value)
                            };
                        else
                        {
                            string errMsg = string.Empty;
                            if (jmr.status != "OK")
                            {
                                errMsg += jmr.status;
                                if (!string.IsNullOrEmpty(jmr.error_message))
                                    errMsg += $" ({jmr.error_message})";
                            }
                            else if (jmr.rows != null
                                && jmr.rows.Length > 0
                                && jmr.rows[0].elements != null
                                && jmr.rows[0].elements.Length > 0)
                            {
                                if (jmr.rows[0].elements[0].status != "OK")
                                    errMsg += $" Route error: {jmr.rows[0].elements[0].status}";
                            }
                            return new GoogleRoute
                            {
                                Status = errMsg
                            };
                        }
                    }
                }
            }
        }

        private string CreateRequestUrl(string origin, string destination)
        {
            string baseUrl = "https://maps.googleapis.com/maps/api/distancematrix/json?units=metric";
            origin = Uri.EscapeDataString(origin);
            destination = Uri.EscapeDataString(destination);
            return $"{baseUrl}&origins={origin}&destinations={destination}&key={apiKey}";
        }

    }
}
