﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Linq;
using GoogleMapClient;

namespace CourierManager
{
    public class Scheduler
    {
        private IGoogleMapApi mapApi;
        private IFareCalculator fareCalculator;
        private IPremiumCalculator premiumCalculator;

        /// <summary>
        /// Injectable constructor
        /// </summary>

        public Scheduler(IGoogleMapApi gma, IFareCalculator fc, IPremiumCalculator pc)
        {
            mapApi = gma;
            fareCalculator = fc;
            premiumCalculator = pc;
            ClearCouriers();
        }

        /// <summary>
        /// The set of couriers registered in the system
        /// </summary>

        public List<Courier> Couriers { get; private set; }

        /// <summary>
        /// Discard the existing set of couriers, ready to inject
        /// our own specific couriers
        /// </summary>

        public void ClearCouriers()
        {
            Couriers = new List<Courier>();
        }

        /// <summary>
        /// Create a new courier at the given starting
        /// location and with the specified name
        /// </summary>
        /// <param name="name">The courier's name</param>
        /// <param name="startLocation">Where the courier starts from</param>
        /// <returns>The new courier object</returns>

        public Courier AddCourier(string name, string startLocation)
        {
            Couriers.Add(new Courier(mapApi, name, startLocation));
            return Couriers.Last();
        }

        /// <summary>
        /// Remove a courier from the set of couriers by name.
        /// NO further journeys will be allocated to this courier.
        /// </summary>
        /// <param name="name">The name of the courier to be removed</param>
        /// <returns>The courier being removed</returns>

        public Courier RemoveCourier(string name)
        {
            Courier removalCandidate = Couriers.FirstOrDefault(c => c.Name == name);
            if (removalCandidate != null)
                Couriers.Remove(removalCandidate);
            return removalCandidate;
        }

        /// <summary>
        /// The list of bookings in order of being placed
        /// </summary>

        public IEnumerable<CourierBooking> Bookings => Couriers.SelectMany(c => c.Schedule);

        /// <summary>
        /// The list of bookings along with their respective assigned couriers
        /// </summary>

        public IEnumerable<BookingSummary> BookingSummaries =>
            Couriers.SelectMany
            (
                c => c.Schedule.Select
                (
                    b => new BookingSummary
                    {
                        CourierName = c.Name, Destination = b.Destination, Distance = b.Distance,
                        Duration = b.Route.Duration, Origin = b.Origin, PickupTime = b.EstimatedPickupTime,
                        Price = b.Fare, ShippingRef = b.ShippingRef, Weight = b.Weight, Id = b.Id
                    }
                )
            );

        /// <summary>
        /// Given a specific booking, find the booking details record that corresponds to it
        /// </summary>
        /// <param name="cb">The courier booking whose details we want</param>
        /// <returns>The details record for the booking</returns>

        public BookingSummary DetailsFromBooking(CourierBooking b)
        {
            var c = Couriers.FirstOrDefault(courier => courier.Schedule.Any(cb => cb == b));
            return new BookingSummary
            {
                CourierName = c.Name,
                Destination = b.Destination,
                Distance = b.Distance,
                Duration = b.Route.Duration,
                Origin = b.Origin,
                PickupTime = b.EstimatedPickupTime,
                Price = b.Fare,
                ShippingRef = b.ShippingRef,
                Weight = b.Weight,
                Id = b.Id
            };
        }

        /// <summary>
        /// Which courier could arrive at a specified location first
        /// </summary>
        /// <param name="location">The desired location</param>
        /// <returns>The nearest courier who could get there first</returns>

        public Courier Closest(string location)
        {
            // return Couriers.OrderBy(c => c.CouldArriveAt(mapApi, location)).First();
            return Couriers.OrderBy(c => c.TimeWhenNextFree).FirstOrDefault();
        }

        /// <summary>
        /// The main user story entry point for making a courier booking
        /// </summary>
        /// <param name="shippingRef">A string meaningful to the booker 
        /// that identifies the shipment</param>
        /// <param name="origin">The pickup point</param>
        /// <param name="destination">The drop-off point</param>
        /// <returns>The booking details</returns>

        public CourierBooking MakeBooking(string shippingRef, string origin, string destination, double weight)
        {
            // First find the nearest courier to the origin, i.e. the one who would be free first

            Courier closestCourier = Closest(origin);

            // Now schedule that courier to take this new booking. First define the route.

            IGoogleRoute route = mapApi.GetRoute(origin, destination);
            if (route.Status != "OK")
                throw new ArgumentException("Route not found: " + route.Status);
            CourierBooking newBooking = new CourierBooking(weight, route, fareCalculator, premiumCalculator);
            closestCourier.TakeBooking(newBooking);
            newBooking.ShippingRef = shippingRef;
            return newBooking;
        }
    }
}
